import { create } from "zustand";
import axiosInstance from "@/utils/axios";
import { AddPassengerToCircuit, AddPassengersToCircuitResponse } from "@/shared/types/circuit";
import { circuitEndpoints } from "../../endpoints/circuit";

type PassengersCircuitStore = {
  loading: boolean;
  error: string | null;
  addedPassengers: AddPassengersToCircuitResponse[];
  addPassengersToCircuit: (payload: AddPassengerToCircuit) => Promise<void>;
  getPassengersOfCircuit: (circuitId: string) => Promise<void>;
};

export const usePassengersCircuitStore = create<PassengersCircuitStore>((set) => ({
  loading: false,
  error: null,
  addedPassengers: [],
  addPassengersToCircuit: async (payload) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.post<AddPassengersToCircuitResponse[]>(
        circuitEndpoints.circuit.addPassengers,
        payload
      );
      set({ 
        loading: false, 
        addedPassengers: response.data 
      });
    } catch (error: any) {
      set({
        error: error.message || "Erreur lors de l'ajout des usagers au circuit",
        loading: false,
      });
      throw error;
    }
  },
  getPassengersOfCircuit: async (circuitId: string) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.get<AddPassengersToCircuitResponse[]>(
        circuitEndpoints.circuit.ammendment.getPassengersOfCircuit(circuitId)
      );
      set({ 
        loading: false, 
        addedPassengers: response.data 
      });
    } catch (error: any) {
      set({
        error: error.message || "Erreur lors de la récupération des usagers du circuit",
        loading: false,
      });
      throw error;
    }
  }
}));