import { create } from 'zustand';
import axiosInstance from '@/utils/axios';
import { calendarEndpoints } from '../../endpoints/hr/calendar/calendar';
import { IDriver } from '@/shared/types/driver';
import { IAgency } from '@/shared/types/infrastructure';

interface CalendarEventResponse {
  id: string;
  employeeType: 'chauffeur' | 'sedentaire';
  type: string;
  employeeId?: string;
  agenceId?: string;
  date: string;
  hour?: string;
  notes?: string;
  notifications: string[];
  isAllDay: boolean;
}

interface CreateEventRequest {
  employeeType: 'chauffeur' | 'sedentaire';
  type: string;
  employeeId?: string;
  agenceId?: string;
  date: string;
  hour?: string;
  notes?: string;
  notifications: string[];
  isAllDay: boolean;
}

interface UpdateEventRequest extends CreateEventRequest {
  id: string;
}

interface CalendarStore {
  events: CalendarEventResponse[];
  loading: boolean;
  error: string | null;
  drivers: IDriver[];
  agencies: IAgency[];
  notificationIntervals: Array<{ value: string; label: string }>;
  
  fetchEvents: () => Promise<void>;
  createEvent: (eventData: CreateEventRequest) => Promise<CalendarEventResponse>;
  updateEvent: (eventData: UpdateEventRequest) => Promise<CalendarEventResponse>;
  deleteEvent: (id: string | number) => Promise<void>;
  fetchDrivers: () => Promise<void>;
  fetchAgencies: () => Promise<void>;
  fetchNotificationIntervals: () => Promise<void>;
}

export const useCalendarStore = create<CalendarStore>((set, get) => ({
  events: [],
  loading: false,
  error: null,
  drivers: [],
  agencies: [],
  notificationIntervals: [],

  fetchEvents: async () => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.get(calendarEndpoints.events.getAll);
      set({ events: response.data, loading: false });
    } catch (error: any) {
      set({ 
        error: error.response?.data?.message || 'Erreur lors du chargement des événements', 
        loading: false 
      });
    }
  },

  createEvent: async (eventData: CreateEventRequest) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.post(calendarEndpoints.events.create, eventData);
      set({ loading: false });
      return response.data;
    } catch (error: any) {
      set({ 
        error: error.response?.data?.message || 'Erreur lors de la création de l\'événement', 
        loading: false 
      });
      throw error;
    }
  },

  updateEvent: async (eventData: UpdateEventRequest) => {
    set({ loading: true, error: null });
    try {
      const response = await axiosInstance.put(calendarEndpoints.events.update(eventData.id), eventData);
      set({ loading: false });
      return response.data;
    } catch (error: any) {
      set({ 
        error: error.response?.data?.message || 'Erreur lors de la mise à jour de l\'événement', 
        loading: false 
      });
      throw error;
    }
  },

  deleteEvent: async (id: string | number) => {
    set({ loading: true, error: null });
    try {
      await axiosInstance.delete(calendarEndpoints.events.delete(id));
      set({ loading: false });
    } catch (error: any) {
      set({ 
        error: error.response?.data?.message || 'Erreur lors de la suppression de l\'événement', 
        loading: false 
      });
      throw error;
    }
  },

  fetchDrivers: async () => {
    try {
      const response = await axiosInstance.get(calendarEndpoints.employees.getDrivers);
      set({ drivers: response.data || [] });
    } catch (error: any) {
      console.error('Erreur lors du chargement des chauffeurs:', error);
      set({ drivers: [] });
    }
  },

  fetchAgencies: async () => {
    try {
      const response = await axiosInstance.get(calendarEndpoints.employees.getAgencies);
      set({ agencies: response.data || [] });
    } catch (error: any) {
      console.error('Erreur lors du chargement des agences:', error);
      set({ agencies: [] });
    }
  },

  fetchNotificationIntervals: async () => {
    try {
      const response = await axiosInstance.get(calendarEndpoints.notifications.getIntervals);
      set({ notificationIntervals: response.data || [] });
    } catch (error: any) {
      console.error('Erreur lors du chargement des intervalles de notification:', error);
      set({ 
        notificationIntervals: [
          { value: '15min', label: '15 minutes avant' },
          { value: '30min', label: '30 minutes avant' },
          { value: '1hour', label: '1 heure avant' },
          { value: '1day', label: '1 jour avant' },
          { value: '1week', label: '1 semaine avant' }
        ] 
      });
    }
  },
})); 