import { memo, useState, useCallback } from 'react';

import Stack from '@mui/material/Stack';
import Collapse from '@mui/material/Collapse';
import ListSubheader from '@mui/material/ListSubheader';

import NavList from './nav-list';
import { NavProps, NavGroupProps } from '../types';
import { bgBlur, noWrapStyles } from '@/shared/theme/css';
import { useTheme } from '@mui/material/styles';
import FontAwesome from '../../fontawesome';
import { faAngleDown, faAngleRight } from '@fortawesome/free-solid-svg-icons';
import { Typography } from '@mui/material';
import CustomTooltip from '../../tooltips/tooltip-custom';

// ----------------------------------------------------------------------

function NavSectionVertical({ data, slotProps, ...other }: NavProps) {
  return (
    <Stack component="nav" id="nav-section-vertical" {...other}>
      <Group
        key={data.title}
        icon={data.icon}
        subheader={data.title}
        items={data.children}
        slotProps={slotProps}
      />
    </Stack>
  );
}
export default memo(NavSectionVertical);

// ----------------------------------------------------------------------

function Group({ subheader, items, icon, slotProps }: NavGroupProps) {
  const [open, setOpen] = useState(true);
  const theme = useTheme();

  const handleToggle = useCallback(() => {
    setOpen((prev) => !prev);
  }, []);

  const renderContent = items?.map((list: any, index) => {
    return (
      <NavList
        key={list.title}
        data={list}
        depth={1}
        slotProps={slotProps}
        isLastParent={index === items.length - 1}
      />
    );
  });

  return (
    <Stack sx={{ px: 2 }}>
      {subheader ? (
        <>
          <ListSubheader
            disableGutters
            disableSticky
            onClick={handleToggle}
            sx={{
              cursor: 'pointer',
              display: 'inline-flex',
              borderRadius: 2,
              color: (theme) => theme.palette.primary.main,
              mb: `${slotProps?.gap || 4}px`,
              p: (theme) => theme.spacing(0.5, 0.8, 0.5, 2),
              ...bgBlur({
                color: theme.palette.primary.main,
                opacity: 0.1,
              }),
              ...slotProps?.subheader,
            }}
          >
            <Stack sx={{ p: (theme) => theme.spacing(0.2, 1, 0.5, 0) }}>{icon}</Stack>
            <CustomTooltip
              PopperProps={{
                modifiers: [
                  {
                    name: 'offset',
                    options: {
                      offset: [0, 8],
                    },
                  },
                ],
              }}
              arrow
              title={subheader}
              placement="right"
            >
              <Typography
                sx={{
                  p: (theme) => theme.spacing(0.2, 0, 0.5, 0),
                  fontSize: 15,
                  fontWeight: theme.typography.fontWeightBold,
                  whiteSpace: 'nowrap',
                  ...noWrapStyles,
                }}
              >
                {subheader}
              </Typography>
            </CustomTooltip>

            <Stack flexGrow={1} direction="row" alignItems="center" justifyContent="flex-end">
              <FontAwesome width={15} icon={open ? faAngleDown : faAngleRight} />
            </Stack>
          </ListSubheader>

          <Collapse in={open}>{renderContent}</Collapse>
        </>
      ) : (
        renderContent
      )}
    </Stack>
  );
}
