"use client"

import { useCallback, useMemo } from "react"
import { Box, Stack } from "@mui/material"
import { HEADER } from "@/layouts/config-layout"
import TableControlBar from "@/shared/components/table/table-control-bar"
import { type TableConfig, useTableManager } from "@/hooks/use-table-manager"
import { type TabConfig, useTabsManager } from "@/hooks/use-tabs-manager"
import { TableType } from "@/shared/types/common"
import TabsCustom from "@/shared/components/tabs/tabs-custom"
import TableManager from "@/shared/components/table/table-manager"
import SuiviTrajetTableRow from "./suivi-trajet-table-row"
import SuiviTrajetForm from "./suivi-trajet-new-view"
import { IRouteTracking } from "@/shared/types/suivi-trajet"
import { _routeTracking, DEFAULT_ROUTE_TRACKING_TABLE_HEAD, INITIAL_ROUTE_TRACKING_DATA } from "@/shared/_mock/_suiviTrajet"

const suiviTrajetConfig: TableConfig<IRouteTracking> = {
  initialData: _routeTracking,
  defaultTableHead: DEFAULT_ROUTE_TRACKING_TABLE_HEAD,
}

const suiviTrajetTabsConfig: TabConfig<IRouteTracking> = {
  type: TableType.SuiviTrajet,
  listTitle: "Liste des trajet",
  newItemTitle: "Nouveau trajet",
  initialData: INITIAL_ROUTE_TRACKING_DATA,
  getItemCode: (suiviTrajet) => suiviTrajet.intitule,
}

export default function SuiviTrajetListView() {
  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
  } = useTabsManager<IRouteTracking>(suiviTrajetTabsConfig)

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    dataToExport,
    handleStatusChange
  } = useTableManager<IRouteTracking>(suiviTrajetConfig)

  const activeTabData = useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab)
  }, [tabs, activeTab])

  const renderRow = useCallback(
    (row: IRouteTracking) => (
      <SuiviTrajetTableRow
        key={row.id}
        dense={table.dense}
        row={row}
        handleEdit={handleEdit}
        handleCopy={handleCopy}
        handleDetails={handleView}
        selected={table.selected.includes(row.id)}
        onSelectRow={() => table.onSelectRow(row.id)}
        columns={tableHead}
        onStatusChange={handleStatusChange}
      />
    ),
    [table, handleEdit, handleCopy, handleView, tableHead, handleStatusChange],
  )

  const isList = activeTab === "list"

  return (
    <Box
      sx={{
        display: "flex",
        flexDirection: "column",
        height: `calc(100vh - ${HEADER.H_DESKTOP + 8}px)`,
      }}
    >
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.SuiviTrajet}
          activeTab={activeTab}
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={dataToExport}
        />
      </Stack>

      <TabsCustom
        type={TableType.SuiviTrajet}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      {isList ? (
        <TableManager
          filteredData={filteredData}
          table={table}
          tableHead={tableHead}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderRow}
        />
      ) : (
       <SuiviTrajetForm
          suiviTrajet={activeTabData?.content as IRouteTracking}
          mode={activeTabData?.mode || "view"}
          onClose={() => handleCancel(activeTab)}
        />
      )}
    </Box>
  )
}
