import { ILeave, ILeaveHistory, LeaveType, LeaveStatus, NotificationMode } from '@/shared/types/leave';

const _firstNames = ["Jean", "Marie", "Pierre", "Sophie", "André", "Philippe", "Catherine", "Michel"];
const _lastNames = ["Dupont", "Martin", "Bernard", "Dubois", "Thomas", "Robert", "Richard", "Petit"];

const _REASONS = [
  "Maladie nécessitant un repos",
  "Congé parental pour naissance",
  "Congés annuels planifiés",
  "Raisons personnelles",
  "RTT conventionnel"
];

const LEAVE_TYPES: LeaveType[] = [
  LeaveType.PAID,
  LeaveType.SICK,
  LeaveType.PARENTAL,
  LeaveType.UNPAID,
  LeaveType.RTT
];

const LEAVE_STATUS: LeaveStatus[] = [
  'En attente',
  'Approuvée',
  'Refusée'
];

const NOTIFICATION_MODES: NotificationMode[] = [
  'SMS',
  'Email',
  'Application'
];

const calculateDays = (startDate: Date, endDate: Date): number => {
  const diffTime = Math.abs(endDate.getTime() - startDate.getTime());
  return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
};

const generateHistory = (
  submissionDate: Date, 
  processDate: Date, 
  status: LeaveStatus,
  employeeName: string
): ILeaveHistory[] => {
  const history: ILeaveHistory[] = [
    {
      date: submissionDate.toISOString().split('T')[0],
      action: 'Soumission',
      user: employeeName,
      comment: "Demande initiale de congé"
    }
  ];

  if (status !== 'En attente') {
    history.push({
      date: processDate.toISOString().split('T')[0],
      action: status === 'Approuvée' ? 'Approbation' : 'Rejet',
      user: _lastNames[Math.floor(Math.random() * _lastNames.length)],
      comment: _REASONS[Math.floor(Math.random() * _REASONS.length)]
    });
  }

  return history;
};

export const _leaves: ILeave[] = [...Array(30)].map((_, index) => {
  const submissionDate = new Date(2024, index % 12, Math.floor(Math.random() * 28) + 1);
  const startDate = new Date(submissionDate);
  startDate.setDate(startDate.getDate() + Math.floor(Math.random() * 14) + 1);
  const endDate = new Date(startDate);
  endDate.setDate(endDate.getDate() + Math.floor(Math.random() * 14) + 1);
  const processDate = new Date(submissionDate);
  processDate.setDate(processDate.getDate() + Math.floor(Math.random() * 3) + 1);

  const employeeName = `${_lastNames[index % _lastNames.length]} ${_firstNames[index % _firstNames.length]}`;

  const status = ['En attente', 'Approuvée', 'Refusée'][Math.floor(Math.random() * 3)] as LeaveStatus;
  
  const leaveType = LEAVE_TYPES[index % LEAVE_TYPES.length];
  const notificationMode = NOTIFICATION_MODES[index % NOTIFICATION_MODES.length];

  return {
    id: (index + 1).toString(),
    employeeName,
    submissionDate: submissionDate.toISOString().split('T')[0],
    leaveType,
    startDate: startDate.toISOString().split('T')[0],
    endDate: endDate.toISOString().split('T')[0],
    status,
    processDate: processDate.toISOString().split('T')[0],
    notificationMode,
    reason: _REASONS[Math.floor(Math.random() * _REASONS.length)],
    daysCount: calculateDays(startDate, endDate),
    employeeId: `EMP-${index + 1}`,
    history: generateHistory(submissionDate, processDate, status, employeeName),
    isActive: Math.random() > 0.5
  } as ILeave;
});

export const INITIAL_LEAVE_DATA: ILeave = {
  id: '',
  employeeName: '',
  submissionDate: new Date().toISOString().split('T')[0],
  leaveType: LeaveType.PAID,
  startDate: new Date().toISOString().split('T')[0],
  endDate: new Date().toISOString().split('T')[0],
  status: 'En attente',
  processDate: new Date().toISOString().split('T')[0],
  notificationMode: 'Application',
  reason: '',
  daysCount: 0,
  employeeId: '',
  history: [],
  isActive: true
};

export const LEAVE_TABLE_COLUMNS = [
  { id: 'employeeName', label: 'Nom & Prénom', type: 'text' },
  { id: 'submissionDate', label: 'Date de soumission', type: 'date' },
  { id: 'leaveType', label: 'Type de congé', type: 'text' },
  { id: 'startDate', label: 'Date de début', type: 'date' },
  { id: 'endDate', label: 'Date de fin', type: 'date' },
  { id: 'daysCount', label: 'Nombre de jours', type: 'number' },
  { id: 'status', label: 'Statut', type: 'status' },
  { id: 'processDate', label: 'Date de traitement', type: 'date' },
  { id: 'notificationMode', label: 'Mode de réception', type: 'text' },
  { id: 'actions', label: 'Actions', type: 'actions' }
];