import { forwardRef } from 'react';

import Box from '@mui/material/Box';
import Link from '@mui/material/Link';
import { alpha, styled } from '@mui/material/styles';
import ListItemButton from '@mui/material/ListItemButton';

import { RouterLink } from '@/routes/router-link';

import { NavItemProps, NavItemStateProps } from '../types';
import { Stack } from '@mui/system';
import FontAwesome from '../../fontawesome';
import { faAngleDown, faAngleRight } from '@fortawesome/free-solid-svg-icons';
import { noWrapStyles } from '@/shared/theme/css';
import CustomTooltip from '../../tooltips/tooltip-custom';

// ----------------------------------------------------------------------

const NavItem = forwardRef<HTMLDivElement, NavItemProps>(
  (
    {
      title,
      path,
      icon,
      //
      open,
      depth,
      active,
      hasChild,
      externalLink,
      ...other
    },
    ref
  ) => {
    const subItem = depth !== 1;

    const renderContent = (
      <StyledNavItem ref={ref} disableGutters open={open} depth={depth} active={active} {...other}>
        {subItem && icon ? (
          <Box component="span" className="icon">
            {icon}
          </Box>
        ) : (
          <Box component="span" className="sub-icon" />
        )}
        {title && (
          <CustomTooltip
            arrow
            title={title}
            placement="right"
            PopperProps={{
              modifiers: [
                {
                  name: 'offset',
                  options: {
                    offset: [0, 8],
                  },
                },
              ],
            }}
          >
            <Box component="span" className="label">
              {title}
            </Box>
          </CustomTooltip>
        )}

        {hasChild && (
          <Stack flexGrow={1} direction="row" alignItems="center" justifyContent="flex-end">
            <FontAwesome
              width={15}
              icon={open ? faAngleDown : faAngleRight}
              sx={{
                transition: 'transform 0.2s',
              }}
            />
          </Stack>
        )}
      </StyledNavItem>
    );

    if (externalLink)
      return (
        <Link href={path} target="_blank" rel="noopener" color="inherit" underline="none">
          {renderContent}
        </Link>
      );

    return (
      <Link component={RouterLink} href={!  hasChild ? path : '#'} color="inherit" underline="none">
        {renderContent}
      </Link>
    );
  }
);
NavItem.displayName = 'NavItem';
export default NavItem;

// ----------------------------------------------------------------------

const StyledNavItem = styled(ListItemButton, {
  shouldForwardProp: (prop) => prop !== 'active',
})<NavItemStateProps>(({ active, open, depth, theme }) => {
  const subItem = depth !== 1;
  const subSubItem = depth && depth > 2;
  const opened = open && !active;

  const baseStyles = {
    item: {
      marginBottom: 4,
      marginTop: 4,
      borderRadius: 8,
      color: theme.palette.secondary.main,
      padding: theme.spacing(0.5, 1, 0.5, 1),
    },
    subItem: {
      marginBottom: 4,
      marginTop: 4,
      borderRadius: 8,
      color: theme.palette.text.secondary,
      padding: theme.spacing(0.5, 1, 0.5, 1),
    },
    label: {
      ...noWrapStyles,
      whiteSpace: 'nowrap',
      ...theme.typography.body2,
      textTransform: 'capitalize',
      fontWeight: active && !subItem ? '600' : '500',
    },
    arrow: {
      flexShrink: 0,
      marginLeft: theme.spacing(0.75),
    },
  } as const;

  return {
    // Root item
    ...(!subItem && {
      ...baseStyles.item,
      minHeight: 36,
      marginLeft: theme.spacing(5),
      '& .sub-icon': {
        display: 'none',
      },
      '& .label': {
        ...baseStyles.label,
      },
      '& .arrow': {
        ...baseStyles.arrow,
      },

      ...(active && {
        color:
          theme.palette.mode === 'light' ? theme.palette.primary.main : theme.palette.primary.light,
        backgroundColor: alpha(theme.palette.primary.main, 0.1),
        '&:hover': {
          backgroundColor: alpha(theme.palette.primary.main, 0.16),
        },
      }),
      ...(opened && {
        backgroundColor: alpha(theme.palette.primary.main, 0.1),
      }),
    }),

    // Sub item
    ...(subItem &&  {
      ...baseStyles.subItem,
      minHeight: 36,
      marginLeft: subSubItem ? theme.spacing(13) : theme.spacing(9),
      '& .label': {
        ...baseStyles.label,
      },
      '& .arrow': {
        ...baseStyles.arrow,
      },

      ...(active && {
        color:
          theme.palette.mode === 'light' ? theme.palette.text.primary : theme.palette.text.primary,
        backgroundColor: alpha(theme.palette.text.secondary, 0.1),
      }),
      ...(opened && {
        backgroundColor: alpha(theme.palette.text.secondary, 0.1),
      }),
    }),
  };
});
