import { ModeType } from '@/shared/types/common';
import * as Yup from 'yup';

export const agentSchema = (mode: ModeType) =>Yup.object().shape({
  lastName: Yup.string()
    .required('Le nom est requis')
    .min(2, 'Le nom doit contenir au moins 2 caractères'),
  firstName: Yup.string()
    .required('Le prénom est requis')
    .min(2, 'Le prénom doit contenir au moins 2 caractères'),
  birthDate: Yup.date()
    .required('La date de naissance est requise')
    .max(new Date(), 'La date de naissance ne peut pas être dans le futur'),
  birthPlace: Yup.string().required('Le lieu de naissance est requis'),
  mapAdresse: Yup.object().shape({
    address: Yup.string().required("L'adresse est requise"),
    lat: Yup.number(),
    lng: Yup.number(),
    city: Yup.string(),
    postalCode: Yup.string(),
    country: Yup.string(),
    formattedAddress: Yup.string().required("L'adresse est requise"),
  }),
  
  phoneNumber: Yup.string()
    .required('Le numéro de téléphone est requis')
    .matches(/^[0-9+\s-]{10,}$/, 'Format de téléphone invalide'),
  email: Yup.string().required("L'email est requis").email("Format d'email invalide"),
  password: mode === 'add' || mode === 'copy'
  ? Yup.string()
      .required('Le mot de passe est requis')
      .min(8, 'Le mot de passe doit contenir au moins 8 caractères')
      .matches(/[A-Z]/, 'Le mot de passe doit contenir au moins une majuscule')
      .matches(/[a-z]/, 'Le mot de passe doit contenir au moins une minuscule')
      .matches(/[0-9]/, 'Le mot de passe doit contenir au moins un chiffre')
  : Yup.string()
      .notRequired()
      .test(
        'password-rules',
        'Le mot de passe doit contenir au moins 8 caractères, une majuscule, une minuscule et un chiffre',
        function (value) {
          if (!value) return true;
          const hasMinLength = value.length >= 8;
          const hasUpperCase = /[A-Z]/.test(value);
          const hasLowerCase = /[a-z]/.test(value);
          const hasNumber = /[0-9]/.test(value);
          return hasMinLength && hasUpperCase && hasLowerCase && hasNumber;
        }
      ),


  socialSecurityNumber: Yup.string().required('Le numéro de sécurité sociale est requis'),
  agentIdentifier: Yup.string().required("L'identifiant agent est requis"),
  agencyName: Yup.string().required("L'agence est requise"),
  departmentName: Yup.string().required('Le département est requis'),
  teams: Yup.array()
    .of(
      Yup.object().shape({
        id: Yup.number().notRequired(),
      })
    ).nullable(),

  contractType: Yup.string().required('Le type de contrat est requis'),
  collaboratorStatus: Yup.string().required('Le statut est requis'),
  roles: Yup.array()
    .of(
      Yup.object({
        id: Yup.string().required("L'identifiant du rôle est requis"),
        name: Yup.string().required('Le nom du rôle est requis'),
        description: Yup.string().notRequired(),
      })
    )
    .min(1, 'Le rôle est requis')
    .required('Le rôle est requis'),
});
