import { useState, useEffect } from 'react';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { TableColumn } from '@/shared/types/common';
import React from 'react';
import type { SectorList, SectorPageFilterParams } from '@/shared/types/infrastructure';
import { useInfrastructureStore } from '@/shared/api/stores/admin-service/infrastructureStore';

export const DEFAULT_TABLE_HEAD: TableColumn[] = [
    { id: 'id', label: 'N°', type: 'text' },
    { id: 'name', label: 'Nom', type: 'text' },
    { id: 'address', label: 'Adresse', type: 'text' },
    { id: 'createdAt', label: 'Date de création', type: 'date' },
    { id: '', label: 'Action' },
];

const sectorConfig: TableConfig<SectorList> = {
    initialData: [],
    defaultTableHead: DEFAULT_TABLE_HEAD,
};

export function useSectorTable() {
    const [params, setParams] = useState<SectorPageFilterParams>({
        page: 0,
        size: 20,
        sortBy: 'id',
        sortDirection: 'desc'
    });

    const { sectorPage, sectorPageTotalElements, loading, error, fetchSectorPage, startExportInfrastructureDataInExcel, exportInfrastructureDataInPdf } = useInfrastructureStore();

    useEffect(() => {
        fetchSectorPage(params);
    }, [params]);

    const tableManager = useTableManager<any>({
        ...sectorConfig,
        initialData: sectorPage || [],
    });

    useEffect(() => {
        if (tableManager.table.rowsPerPage > 0) {
            setParams(prev => ({
                ...prev,
                page: tableManager.table.page,
                size: tableManager.table.rowsPerPage,
            }));
        }
    }, [tableManager.table.page, tableManager.table.rowsPerPage]);

    const isValidDateInput = (value: any): boolean => {
        if (!value || typeof value !== 'string') return false;
        const date = new Date(value);
        return date instanceof Date && !isNaN(date.getTime());
    };
    
    const isValidFilterValue = (value: any): boolean => {
        return value !== null && value !== undefined && value !== '';
    };

    useEffect(() => {
        const baseParams: SectorPageFilterParams = {
            page: tableManager.table.page || 0,
            size: tableManager.table.rowsPerPage || 20,
            sortBy: tableManager.table.orderBy || 'id',
            sortDirection: (tableManager.table.order || 'desc').toLowerCase() as 'asc' | 'desc'
        };

        const filterParams: Record<string, string | number> = {};

        Object.entries(tableManager.filters).forEach(([key, { value }]) => {
            if (key === 'createdAt' && isValidDateInput(value)) {
                const date = new Date(value);
                const year = date.getFullYear();
                const month = String(date.getMonth() + 1).padStart(2, '0');
                const day = String(date.getDate()).padStart(2, '0');
                filterParams[key] = `${year}-${month}-${day}T00:00:00`;
            } else if (isValidFilterValue(value)) {
                filterParams[key] = value;
            }
        });

        const newParams: SectorPageFilterParams = {
            ...baseParams,
            ...filterParams
        };

        const paramsChanged = JSON.stringify(newParams) !== JSON.stringify(params);
        if (paramsChanged) {
            setParams(newParams);
        }
    }, [tableManager.filters, tableManager.table.orderBy, tableManager.table.order]);    

    const handlePageChange = (event: React.MouseEvent<HTMLButtonElement> | null, newPage: number) => {
        setParams(prev => ({
            ...prev,
            page: newPage
        }));
    };

    const handleRowsPerPageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const size = parseInt(event.target.value, 10);
        if (!isNaN(size) && size > 0) {
            setParams(prev => ({
                ...prev,
                size,
                page: 0
            }));
        }
    };

    const handleStartExportInExcel = () => {
        startExportInfrastructureDataInExcel(params);
    };

    const handleExportInPdf = () => {
        exportInfrastructureDataInPdf(params);
    };

    return {
        ...tableManager,
        loading,
        error,
        totalSectors: sectorPageTotalElements || 0,
        onPageChange: handlePageChange,
        page: params.page,
        onRowsPerPageChange: handleRowsPerPageChange,
        rowsPerPage: params.size,
        sortBy: params.sortBy || 'id',
        sortDirection: params.sortDirection || 'desc',
        filteredData: sectorPage,
        handleStartExportInExcel,
        handleExportInPdf
    };
}