import { Box, Chip, Stack, Typography } from '@mui/material';
import { styled } from '@mui/material/styles';
import { WeekType, DaysOfWeek } from '@/shared/types/passenger';
import { WeekTypeSchedule } from '@/shared/types/circuit';
import { PeriodOfDay } from '@/shared/types/ammendment-passenger';
import { DAYS_CONFIG, PERIODS_CONFIG, WEEK_TYPES } from '@/shared/components/form/transport-schedule-field';

const DayChip = styled(Chip)(({ theme }) => ({
  width: 20,
  height: 20,
  borderRadius: '50%',
  fontSize: '0.55rem',
  '& .MuiChip-label': {
    padding: 0,
  },
  '&.active': {
    backgroundColor: theme.palette.primary.main,
    color: theme.palette.primary.contrastText,
  },
  '&.inactive': {
    backgroundColor: 'transparent',
    border: `1px solid ${theme.palette.primary.main}`
  }
}));

interface JoursAttribuesGridProps {
  weeklyAssignmentSchedule: WeekTypeSchedule[];
}

export default function JoursAttribuesGrid({ weeklyAssignmentSchedule }: JoursAttribuesGridProps) {
  const getDayLabel = (day: DaysOfWeek) => 
    DAYS_CONFIG.find(d => d.key === day)?.label || '';
    
  const getPeriodLabel = (period: PeriodOfDay) => 
    PERIODS_CONFIG.find(p => p.key === period)?.label || '';

  const getWeekTypeLabel = (weekType: WeekType) => 
    WEEK_TYPES.find(w => w.key === weekType)?.label || '';

  const isPeriodActive = (weekType: WeekType, day: DaysOfWeek, period: PeriodOfDay) => {
    const weekSchedule = weeklyAssignmentSchedule.find(w => w.weekType === weekType);
    return weekSchedule?.periods.some(
      p => p.dayOfWeek === day && p.periodOfDay === period
    ) ?? false;
  };

  const renderDayPeriod = (weekType: WeekType, day: DaysOfWeek) => (
    <Stack direction="row" spacing={0.3} alignItems="center">
      <DayChip
        label={`${getDayLabel(day)}${getPeriodLabel(PeriodOfDay.MORNING)}`}
        className={isPeriodActive(weekType, day, PeriodOfDay.MORNING) ? 'active' : 'inactive'}
        size="small"  
      />
      <DayChip
        label={`${getDayLabel(day)}${getPeriodLabel(PeriodOfDay.EVENING)}`}
        className={isPeriodActive(weekType, day, PeriodOfDay.EVENING) ? 'active' : 'inactive'}
        size="small"
      />
    </Stack>
  );

  return (
    <Box sx={{ py: 1.5 }}>
      <Stack spacing={1.5}>
        {WEEK_TYPES.map(weekType => (
          <Box key={weekType.key}>
            <Stack direction="row" alignItems="center" spacing={1}>
              <Typography sx={{ fontSize: '9px', width: 30, textAlign: weekType.key === WeekType.EVEN ? 'left' : undefined }}>
                {weekType.label}
              </Typography>
              <Stack direction="row" spacing={1} justifyContent="space-between" flexGrow={1}>
                {DAYS_CONFIG.map(day => (
                  <Box key={`${weekType.key}-${day.key}`}>
                    {renderDayPeriod(weekType.key, day.key)}
                  </Box>
                ))}
              </Stack>
            </Stack>
          </Box>
        ))}
      </Stack>
    </Box>
  );
}