import { useState } from 'react';
import { 
  Dialog, 
  DialogTitle, 
  DialogContent, 
  DialogActions, 
  Button,
  Typography,
  Box,
  CircularProgress,
  Stack,
  useTheme,
  IconButton
} from '@mui/material';
import DownloadIcon from '@mui/icons-material/Download';
import SendIcon from '@mui/icons-material/Send';
import CloseIcon from '@mui/icons-material/Close';
import { AddButton, outlinedColor } from '@/shared/components/table/styles';

interface ContractGenerationDialogProps {
  open: boolean;
  onClose: () => void;
  onGenerate: () => Promise<void>;
  onSendToDocuSign: () => Promise<void>;
  candidatName: string;
}

export default function ContractGenerationDialog({ 
  open, 
  onClose, 
  onGenerate, 
  onSendToDocuSign,
  candidatName 
}: ContractGenerationDialogProps) {
  const theme = useTheme();
  const [isGenerating, setIsGenerating] = useState(false);
  const [isSending, setIsSending] = useState(false);

  const handleGenerate = async () => {
    setIsGenerating(true);
    try {
      await onGenerate();
    } finally {
      setIsGenerating(false);
    }
  };

  const handleSendToDocuSign = async () => {
    setIsSending(true);
    try {
      await onSendToDocuSign();
    } finally {
      setIsSending(false);
    }
  };

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="sm" 
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 2,
          boxShadow: theme.shadows[10]
        }
      }}
    >
      <DialogTitle sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        padding: theme.spacing(1, 2, 2)
      }}>
        <Typography variant="h6" fontWeight="medium">
          Générer contrat
        </Typography>
        <IconButton
          onClick={onClose} 
          disabled={isGenerating || isSending}
        >
          <CloseIcon />
        </IconButton>
      </DialogTitle>
      
      <DialogContent sx={{ padding: theme.spacing(3) }}>
        <Typography variant="body1" color="text.secondary" paragraph>
          Vous êtes sur le point de générer le contrat pour le candidat{' '}
          <Typography component="span" fontWeight="medium" color="text.primary">
            {candidatName}
          </Typography>.
          Veuillez choisir une action ci-dessous :
        </Typography>
        
        <Stack 
          direction={{ xs: 'column', sm: 'row' }} 
          spacing={2} 
          sx={{ 
            mt: 3,
            justifyContent: 'center',
            alignItems: 'center'
          }}
        >          
          <Button
            variant="outlined"
            size="large"
            color="secondary"
            startIcon={isSending ? <CircularProgress size={20} color="inherit" /> : <SendIcon />}
            onClick={handleSendToDocuSign}
            disabled={isGenerating || isSending}
            sx={{
              py: 1.5,
              ...outlinedColor
            }}
            fullWidth
          >
            Envoyer via DocuSign
          </Button>
          <AddButton
            variant="contained"
            size="large"
            startIcon={isGenerating ? <CircularProgress size={20} color="inherit" /> : <DownloadIcon />}
            onClick={handleGenerate}
            disabled={isGenerating || isSending}
            sx={{py: 1.5}}
            fullWidth
          >
            Télécharger PDF
          </AddButton>
        </Stack>
      </DialogContent>
      
    </Dialog>
  );
}