import { _noms, _prenoms } from ".";

export type EmployeeEventType = 'contract_end' | 'trial_end' | 'document_expiry' | 'custom';

// Mock data for drivers
export const MOCK_DRIVERS = [
  { value: 'chauf1', label: 'Jean Dupont' },
  { value: 'chauf2', label: 'Marie Martin' },
  { value: 'chauf3', label: 'Pierre Dubois' },
  { value: 'chauf4', label: 'Sophie Bernard' },
  { value: 'chauf5', label: 'Lucas Moreau' }
] as const;

// Mock data for agencies
export const MOCK_AGENCIES = [
  { value: 'paris', label: 'Paris' },
  { value: 'lyon', label: 'Lyon' },
  { value: 'marseille', label: 'Marseille' }
] as const;

// Mock data for driver statuses
export const MOCK_DRIVER_STATUSES = [
  { value: 'active', label: 'Actif', color: '#54D62C' },
  { value: 'inactive', label: 'Inactif', color: '#FF4842' },
  { value: 'vacation', label: 'En congé', color: '#FFC107' }
] as const;

export interface EmployeeEvent {
  id: string;
  title: string;
  employeeId: string;
  employeeName: string;
  start: string;
  end: string;
  type: EmployeeEventType;
  color: string;
  description?: string;
  allDay: boolean;
  status: 'pending' | 'toRenew' | 'expired' | 'renewed';
  documents?: Array<{ name: string; type: string; url: string; }>;
  notifications?: string[];
  agency: 'paris' | 'lyon' | 'marseille';
  driverStatus: 'active' | 'inactive' | 'vacation';
  comments?: Array<{
    author: string;
    avatar?: string;
    date: string;
    text: string;
  }>;
}

// Color mapping for different event types
export const EVENT_COLORS = {
  contract_end: '#EF7564',
  trial_end: '#5BA4CF',
  document_expiry: '#FFB020',
  custom: '#B7B7B7'
};

// Fixed set of events
// export const _employeeEvents: EmployeeEvent[] = [...]

// Initial event data structure
export const INITIAL_EVENT_DATA: EmployeeEvent = {
  id: '',
  employeeId: '',
  employeeName: '',
  title: '',
  start: '',
  end: '',
  type: 'custom',
  color: EVENT_COLORS.custom,
  description: '',
  allDay: false,
  status: 'pending',
  agency: 'paris',
  driverStatus: 'active',
};

const EMPLOYEES = {
  'emp1': 'Marie Laurent',
  'emp2': 'Pierre Durand',
  'emp3': 'Sophie Martin',
  'emp4': 'Jean Dubois',
  'emp5': 'Claire Rousseau'
};

const EVENT_TYPES: EmployeeEventType[] = ['contract_end', 'trial_end', 'document_expiry', 'custom'];
const EVENT_STATUSES: ('pending' | 'toRenew' | 'expired' | 'renewed')[] = [
  'pending',
  'toRenew',
  'expired',
  'renewed'
];

const NOTIFICATION_OPTIONS = ['1h', '1d', '1w']; // Example notification options

function randomInt(min: number, max: number): number {
  return Math.floor(Math.random() * (max - min + 1)) + min;
}

function randomElement<T>(array: readonly T[] | T[]): T {
  return array[Math.floor(Math.random() * array.length)];
}

function generateRandomEvents(numberOfEvents: number): EmployeeEvent[] {
  const AGENCIES = ['paris', 'lyon', 'marseille'] as const;
  const DRIVER_STATUSES = ['active', 'inactive', 'vacation'] as const;

  const events: EmployeeEvent[] = [];
  const employeeIds = Object.keys(EMPLOYEES);

  for (let i = 0; i < numberOfEvents; i++) {
    // Random date within -30 to +60 days from now
    const startDaysOffset = randomInt(-30, 60);
    const startDate = new Date();
    startDate.setDate(startDate.getDate() + startDaysOffset);
    
    // Random duration 1-5 days for all-day events, or 1-8 hours for non-all-day events
    const isAllDay = Math.random() > 0.3; // 70% chance of all-day event
    let endDate = new Date(startDate);
    
    if (isAllDay) {
      endDate.setDate(startDate.getDate() + randomInt(1, 5));
    } else {
      startDate.setHours(randomInt(9, 16), 0, 0, 0); // Start between 9 AM and 4 PM
      endDate = new Date(startDate);
      endDate.setHours(startDate.getHours() + randomInt(1, 8)); // 1-8 hours duration
    }

    const employeeId = randomElement(employeeIds);
    const type = randomElement(EVENT_TYPES) as EmployeeEventType;
    const status = randomElement(EVENT_STATUSES);

    const event: EmployeeEvent = {
      id: `event-${i + 1}`,
      employeeId,
      employeeName: EMPLOYEES[employeeId as keyof typeof EMPLOYEES],
      type,
      title: `${type === 'contract_end' ? 'Fin de contrat' : 
              type === 'trial_end' ? "Fin de période d'essai" :
              type === 'document_expiry' ? 'Expiration de documents' :
              'Événement personnalisé'} - ${EMPLOYEES[employeeId as keyof typeof EMPLOYEES]}`,
      start: startDate.toISOString(),
      end: endDate.toISOString(),
      color: EVENT_COLORS[type],
      allDay: isAllDay,
      status,
      description: type === 'custom' ? `Événement personnalisé pour ${EMPLOYEES[employeeId as keyof typeof EMPLOYEES]}` : undefined,
      notifications: Math.random() > 0.5 ? 
        [randomElement(NOTIFICATION_OPTIONS)] : 
        [],
      agency: randomElement(AGENCIES),
      driverStatus: randomElement(DRIVER_STATUSES),
    };

    events.push(event);
  }

  // Sort events by start date
  return events.sort((a, b) => new Date(a.start).getTime() - new Date(b.start).getTime());
}

// Generate 50 random events
export const _employeeEvents = generateRandomEvents(50); 