import { Box, Typography, styled } from '@mui/material';
import { format } from 'date-fns';
import { fr } from 'date-fns/locale';
import CustomPopover from '@/shared/components/custom-popover';
import { faSun } from '@fortawesome/free-solid-svg-icons';
import FontAwesome from '@/shared/components/fontawesome';
import { CalendarEvent } from '@/hooks/use-calendar';

const PopoverHeader = styled(Typography)(({ theme }) => ({
  padding: theme.spacing(1, 2),
  fontWeight: theme.typography.fontWeightBold,
  fontSize: 16,
}));

const PeriodItem = styled(Box)(({ theme }) => ({
  padding: theme.spacing(1),
  margin: theme.spacing(1),
  borderRadius: theme.shape.borderRadius,
  fontSize: 14,
}));

interface PeriodPopoverProps {
  open: HTMLElement | null;
  onClose: () => void;
  date: Date;
  periods: CalendarEvent[];
}

export function PeriodPopover({ open, onClose, date, periods }: PeriodPopoverProps) {
  const formattedDate = format(date, 'd MMMM', { locale: fr });

  return (
    <CustomPopover
      arrow="top-right"
      open={open}
      onClose={onClose}
      sx={{
        mt: 1,
        width: 250,
        maxHeight: 400,
        overflow: 'auto',
        borderRadius: 2,
      }}
    >
      <PopoverHeader>{formattedDate}</PopoverHeader>
      <Box sx={{ px: 1 }}>
        {periods.map((period) => (
          <PeriodItem
            key={period.id}
            sx={{
              backgroundColor: period.bgColor,
              color: period.color,
              display: 'flex',
              alignItems: 'center',
              gap: 1,
            }}
          >
            <FontAwesome icon={faSun} width={10} />
            {period.label}
          </PeriodItem>
        ))}
      </Box>
    </CustomPopover>
  );
}
