import { Box, Select, MenuItem } from '@mui/material';
import { useTheme } from '@mui/material/styles';

export interface StatusOption {
  value: string;
  label: string;
  color: string | ((theme: any) => string);
  backgroundColor: string | ((theme: any) => string);
}

interface StatusSelectProps {
  value: string;
  options: StatusOption[];
  onChange: (value: string) => void;
  chipComponent?: React.ComponentType<any>;
  fullWidth?: boolean;
}

export default function StatusSelect({ 
  value, 
  options, 
  onChange,
  chipComponent: ChipComponent,
  fullWidth = false
}: StatusSelectProps) {
  const theme = useTheme();
  const currentOption = options.find(opt => opt.value === value);
  
  if (!currentOption) {
    console.warn(`Value ${value} not found in options`);
  }

  const handleClick = (event: React.MouseEvent) => {
    event.stopPropagation(); 
  };

  const getColor = (color: string | ((theme: any) => string)) => {
    return typeof color === 'function' ? color(theme) : color;
  };

  const getBackgroundColor = (bgColor: string | ((theme: any) => string)) => {
    return typeof bgColor === 'function' ? bgColor(theme) : bgColor;
  };

  return (
    <Box sx={{ 
      display: 'inline-flex', 
      alignItems: 'center',
      width: fullWidth ? '100%' : 'auto'
    }}>
      <Select
        value={value || ''} 
        onClick={handleClick}
        size="small"
        onChange={(e) => {
          e.stopPropagation();
          onChange(e.target.value);
        }}
        renderValue={(selected) => {
          const option = currentOption || options[0];
          return ChipComponent ? (
            <ChipComponent
              label={option.label}
              status={option.value}
              theme={theme}
              size="small"
              sx={{
                width: '100%',
                height: '32px',
                padding: '9px 0',
                borderRadius: '8px',
                fontWeight: 500,
                color: getColor(option.color),
                backgroundColor: getBackgroundColor(option.backgroundColor),
                '&:hover': {
                  opacity: 0.8
                }
              }}
            />
          ) : (
            <Box
              sx={{
                width: '100%',
                px: 2,
                py: 0.5,
                borderRadius: '8px',
                fontSize: '0.875rem',
                fontWeight: 500,
                color: getColor(option.color),
                backgroundColor: getBackgroundColor(option.backgroundColor)
              }}
            >
              {option.label}
            </Box>
          );
        }}
        sx={{ 
          width: fullWidth ? '100%' : '120px',
          height: '32px',
          '& .MuiSelect-select': {
            padding: '4px',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center'
          },
          '& .MuiOutlinedInput-notchedOutline': {
            border: 'none'
          }
        }}
        MenuProps={{
          PaperProps: {
            sx: {
              mt: 1,
              '& .MuiList-root': {
                py: 1
              },
              '& .MuiMenuItem-root': {
                px: 1,
                my: 0.5,
                '&:first-of-type': {
                  mt: 0
                },
                '&:last-child': {
                  mb: 0
                }
              }
            }
          }
        }}
      >
        {options.map((option) => (
          <MenuItem 
            key={option.value} 
            value={option.value}
            sx={{ 
              display: 'flex', 
              justifyContent: 'center',
              width: '100%',
              py: 0.5
            }}
          >
            {ChipComponent ? (
              <ChipComponent
                label={option.label}
                status={option.value}
                theme={theme}
                size="small"
                sx={{
                  width: '100%',
                  height: '32px',
                  padding: '9px 0',
                  borderRadius: '8px',
                  fontWeight: 500,
                  color: getColor(option.color),
                  backgroundColor: getBackgroundColor(option.backgroundColor),
                }}
              />
            ) : (
              <Box
                sx={{
                  width: '100%',
                  px: 2,
                  py: 0.5,
                  borderRadius: '8px',
                  fontSize: '0.875rem',
                  fontWeight: 500,
                  color: getColor(option.color),
                  backgroundColor: getBackgroundColor(option.backgroundColor)
                }}
              >
                {option.label}
              </Box>
            )}
          </MenuItem>
        ))}
      </Select>
    </Box>
  );
}