import type React from "react"
import { Box, Typography, Grid, Chip, useTheme } from "@mui/material"
import SvgColor from "@/shared/components/svg-color"
import FontAwesome from "@/shared/components/fontawesome"
import { faClock, faCar, faUser, faRoad, faMapMarkerAlt, faBuilding, faMapPin } from "@fortawesome/free-solid-svg-icons"
import { containerStyles, InfoItem } from "./utils/journey-summary-utils"
import { GeolocStatus } from "@/shared/types/geoloc"
import { GeolocStatusChip, getGeolocStatusLabel } from "@/shared/sections/geolocalisation/styles"


interface VehicleInfo {
  vehicleId: string;
  vehicleName: string;
  driverName: string;
}

interface JourneySummaryProps {
  distance: string
  duration: string
  departDate?: string
  departTime?: string
  arrivalDate?: string
  arrivalTime?: string
  trafficDate?: string
  trafficTime?: string
  isDrawerElement?: boolean
  vehicleInfo?: VehicleInfo
  departureAddress?: string
  arrivalAddress?: string
  agency?: string
  department?: string
  sector?: string
  averageSpeed?: string
  consumption?: string
  status?: string
  statusCode?: GeolocStatus
  showAllInfo?: boolean
  onStatusChange?: (status: string) => void
  hideTitle?: boolean
  useNewLayout?: boolean
}


export const RouteSummary: React.FC<JourneySummaryProps> = ({
  distance,
  duration,
  departDate,
  departTime,
  arrivalDate,
  arrivalTime,
  trafficDate,
  trafficTime,
  isDrawerElement = false,
  vehicleInfo,
  departureAddress,
  arrivalAddress,
  agency,
  department,
  sector,
  averageSpeed,
  consumption,
  status,
  statusCode,
  showAllInfo = false,

  hideTitle = false,
  useNewLayout = false,
}) => {
  const isCompleted = statusCode === GeolocStatus.COMPLETED || status === 'Terminé';
  const theme = useTheme();
  const displayDuration = isCompleted ? duration : '-';
  const displayAverageSpeed = isCompleted ? averageSpeed : '-';
  const displayConsumption = isCompleted ? consumption : '-';
  const displayArrivalDate = isCompleted ? arrivalDate || '-' : '-';
  const displayArrivalTime = isCompleted ? arrivalTime || '-' : '-';

  return (
    <Box sx={{ ...containerStyles(showAllInfo)}}>
      {!hideTitle && (
        <Typography variant="h6" sx={{ mb: 2 }}>
          Résumé du trajet
        </Typography>
      )}
      {useNewLayout ? (
        <Box sx={{ width: "100%" }}>

          <Grid container spacing={2} sx={{ mb: 2 }}>
            <Grid item xs={12} sm={6} md={3}>
              <InfoItem
                icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
                label="Date départ :"
                value={departDate || trafficDate || '-'}
              />
            </Grid>
            <Grid item xs={12} sm={6} md={3}>
              <InfoItem
                icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
                label="Heure départ :"
                value={departTime || trafficTime || '-'}
              />
            </Grid>
            <Grid item xs={12} sm={6} md={3}>
              <InfoItem
                icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
                label="Date d'arrivée :"
                value={displayArrivalDate}
              />
            </Grid>
            <Grid item xs={12} sm={6} md={3}>
              <InfoItem
                icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
                label="Heure d'arrivée :"
                value={displayArrivalTime}
              />
            </Grid>
          </Grid>
          <Grid container spacing={2}>
            {vehicleInfo && (
              <>
                <Grid item xs={12} sm={6} md={3}>
                  <InfoItem
                    icon={<FontAwesome icon={faUser} color="primary.main" width={16} />}
                    label="Chauffeur :"
                    value={vehicleInfo.driverName}
                  />
                </Grid>
                <Grid item xs={12} sm={6} md={3}>
                  <InfoItem
                    icon={<FontAwesome icon={faCar} color="primary.main" width={16} />}
                    label="Véhicule :"
                    value={`${vehicleInfo.vehicleId}`}
                  />
                </Grid>
              </>
            )}
            <Grid item xs={12} sm={6} md={3}>
              <InfoItem
                icon={
                  <SvgColor
                    src="/assets/icons/menu/ic_exploitation.svg"
                    sx={{ color: "primary.main", width: 16, height: 16 }}
                  />
                }
                label="Distance :"
                value={distance}
              />
            </Grid>
            <Grid item xs={12} sm={6} md={3}>
              <InfoItem
                icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
                label="Durée :"
                value={displayDuration}
              />
            </Grid>
          </Grid>
        </Box>
      ) : (
        <Box
          sx={{
            display: "flex",
            gap: 2,
            width: "100%",
            justifyContent: isDrawerElement ? "space-between" : { xs: "space-between", md: "space-around" },
            flexWrap: "wrap",
          }}
        >
          <InfoItem
            icon={
              <SvgColor
                src="/assets/icons/menu/ic_exploitation.svg"
                sx={{ color: "primary.main", width: 16, height: 16 }}
              />
            }
            label="Distance :"
            value={distance}
          />
          <InfoItem
            icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
            label="Durée :"
            value={displayDuration}
          />
          <InfoItem
            icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
            label="Date départ :"
            value={departDate || trafficDate || '-'}
          />
          <InfoItem
            icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
            label="Heure départ :"
            value={departTime || trafficTime || '-'}
          />
          <InfoItem
            icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
            label="Date d'arrivée :"
            value={displayArrivalDate}
          />
          <InfoItem
            icon={<FontAwesome icon={faClock} color="primary.main" width={16} />}
            label="Heure d'arrivée :"
            value={displayArrivalTime}
          />
          
          {vehicleInfo && (
            <>
              <InfoItem
                icon={<FontAwesome icon={faCar} color="primary.main" width={16} />}
                label="Véhicule :"
                value={`${vehicleInfo.vehicleId} - ${vehicleInfo.vehicleName}`}
              />
              <InfoItem
                icon={<FontAwesome icon={faUser} color="primary.main" width={16} />}
                label="Chauffeur :"
                value={vehicleInfo.driverName}
              />
            </>
          )}
        </Box>
      )}
      {showAllInfo && (
        <Grid container spacing={2} sx={{ mt: 2 }}>
          <Grid item xs={12} md={4}>
            <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 'bold' }}>
              Adresses
            </Typography>
            <Box sx={{ mt: 1 }}>
              <InfoItem
                icon={<FontAwesome icon={faMapMarkerAlt} color="primary.main" width={16} />}
                label="Départ :"
                value={departureAddress || '-'}
              />
              <InfoItem
                icon={<FontAwesome icon={faMapPin} color="primary.main" width={16} />}
                label="Arrivée :"
                value={arrivalAddress || '-'}
              />
            </Box>
          </Grid>
          
          <Grid item xs={12} md={4}>
            <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 'bold' }}>
              Organisation
            </Typography>
            <Box sx={{ mt: 1 }}>
              <InfoItem
                icon={<FontAwesome icon={faBuilding} color="primary.main" width={16} />}
                label="Agence :"
                value={agency || '-'}
              />
              <InfoItem
                icon={<FontAwesome icon={faMapPin} color="primary.main" width={16} />}
                label="Département :"
                value={department || '-'}
              />
              <InfoItem
                icon={<FontAwesome icon={faMapPin} color="primary.main" width={16} />}
                label="Secteur :"
                value={sector || '-'}
              />
            </Box>
          </Grid>
          
          <Grid item xs={12} md={4}>
            <Typography variant="subtitle1" sx={{ mb: 2, fontWeight: 'bold' }}>
              Performance
            </Typography>
            <Box sx={{ mt: 1 }}>
              <InfoItem
                icon={<FontAwesome icon={faRoad} color="primary.main" width={16} />}
                label="Vitesse moyenne :"
                value={displayAverageSpeed || '-'}
              />
              <InfoItem
                icon={<FontAwesome icon={faCar} color="primary.main" width={16} />}
                label="Consommation :"
                value={displayConsumption || '-'}
              />
              {status && (
                <Box sx={{ display: 'flex', alignItems: 'center', mt: 1 }}>
                  <FontAwesome icon={faClock} color="primary.main" width={16} />
                  <Typography variant="body2" sx={{ ml: 1, minWidth: '100px',color:'primary.main' }}>
                    Statut :
                  </Typography>
                  <GeolocStatusChip 
                theme={theme}
                label={getGeolocStatusLabel(status as GeolocStatus)}
                status={getGeolocStatusLabel(status as GeolocStatus)}
              />
                </Box>
              )}
            </Box>
          </Grid>
        </Grid>
      )}
    </Box>
  )
}