import { useState, useEffect, useCallback } from 'react';

import Collapse from '@mui/material/Collapse';

import { usePathname } from '@/hooks';
import { useActiveLink } from '@/hooks';

import NavItem from './nav-item';
import { NavListProps, NavSubListProps } from '../types';
import { alpha, Box } from '@mui/system';
import { bgBlur } from '@/shared/theme/css';
import { useTheme } from '@mui/material/styles';

// ----------------------------------------------------------------------

export default function NavList({ data, depth, slotProps,isFirstItem, isLastItem, isLastParent }: NavListProps & { isLastItem?: boolean  ,isFirstItem?: boolean  ,isLastParent?: boolean }) {
  const pathname = usePathname();
  const theme = useTheme();
  const subItem = depth !== 1;
  const active = useActiveLink(data.path, !!data.children);
  const [openMenu, setOpenMenu] = useState(active);

  useEffect(() => {
    if (!active) {
      handleCloseMenu();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pathname]);

  const handleToggleMenu = useCallback(() => {
    if (data.children) {
      setOpenMenu((prev) => !prev);
    }
  }, [data.children]);

  const handleCloseMenu = useCallback(() => {
    setOpenMenu(false);
  }, []);

  return (
    <>
      <NavItem
        open={openMenu}
        onClick={handleToggleMenu}
        //
        title={data.title}
        path={data.path}
        icon={data.icon}
        //
        depth={depth}
        hasChild={!!data.children}
        externalLink={data.path.includes('http')}
        //
        active={active}
        className={`${data.className || ''} ${active ? 'active' : ''}`}
        sx={{
          mb: `${slotProps?.gap}px`,
          ...(depth === 1 ? slotProps?.rootItem : slotProps?.subItem),
          color: 'text.white',
          ...(!isLastParent && subItem && {
            '&::before': {
              content: '""',
              position: "absolute",
              left: (!data.children) ? -46 : (depth > 2 ? -46 - (depth - 2) * 20 : -46),
              top: -1,
              height: !isFirstItem ? "115%" : '155%',
              width: 2,
              ...bgBlur({
                color: theme.palette.text.secondary,
                opacity: 0.1,
              }),            
              transform: "translateY(-50%)",
            },
          }),
          "&::after": {
            content: '""',
            position: "absolute",
            left: -14,
            top: isFirstItem ? -10 : -25,
            zIndex:-1,
            height:isFirstItem ? 32 : 47,
            width: 15,
            border: `2px solid ${alpha(theme.palette.text.secondary,0.1)} `,
            borderTop: "none",
            borderRight: "none",
            borderRadius: "0 0 0 10px",
          },
        
        }} />
   
      {!!data.children && (
        <Collapse in={openMenu} unmountOnExit>
          <NavSubList data={data.children} isLastParent={isLastParent || isLastItem}  depth={depth} slotProps={slotProps} />
        </Collapse>
      )}
    </>
  );
}

// ----------------------------------------------------------------------


function NavSubList({ data, depth, slotProps,isLastParent }: NavSubListProps & { isLastParent?: boolean}) {
  return (
    <Box 
    sx={{
      position: 'relative',
      ...(depth > 1 && {
        '&::before': {
          content: '""',
          position: "absolute",
          left: 26,
          top: -24,
          height: '120%',
          width: 2,
          backgroundColor:(theme) => alpha(theme.palette.text.secondary, 0.1),
        }
      })
    }}>
      {data.map((list,index) => (
        <NavList key={list.title} data={list} depth={depth + 1} isFirstItem={index === 0} isLastItem={index === data.length - 1} isLastParent={isLastParent} slotProps={slotProps} />
      ))}
    </Box>
  );
}
