'use client';

import { Box, styled, SxProps, Theme } from '@mui/system';
import { StyledTab } from '@/shared/theme/css';
import { Tabs } from '@mui/material';
import { useState } from 'react';
import FontAwesome from '../fontawesome';
import { AddButton, ColumnButton, lightColor } from '../table/styles';
import { faArrowLeft, faArrowRight } from '@fortawesome/free-solid-svg-icons';

interface Tab {
  label: string;
  content: React.ReactNode;
  disabled?: boolean;
}

interface TabbedFormProps {
  tabs: Tab[];
  showNavigationButtons?: boolean;
  sx?: SxProps<Theme>;
  hasBorder?: boolean
  isTransparent?: boolean
}

const FormStyledTab = styled(StyledTab)(({ theme, disabled }) => ({
  borderBottom: `2px solid ${disabled ? theme.palette.text.disabled : theme.palette.primary.main}`,
  minWidth: 100,
  [theme.breakpoints.down('lg')]: {
    minWidth: 80,
  },
}));

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function CustomTabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;

  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`tabpanel-${index}`}
      aria-labelledby={`tab-${index}`}
      {...other}
    >
      {value === index && <Box sx={{ pt: 3 }}>{children}</Box>}
    </div>
  );
}

export default function TabbedForm({ tabs, showNavigationButtons, sx, hasBorder, isTransparent }: TabbedFormProps) {
  const [value, setValue] = useState(0);

  const handleChange = (event: React.SyntheticEvent, newValue: number) => {
    setValue(newValue);
  };
  const handlePrevious = () => {
    setValue((prev) => Math.max(0, prev - 1));
  };
  const handleNext = () => {
    setValue((prev) => Math.min(tabs.length - 1, prev + 1));
  };
  return (
    <Box sx={{ width: "100%", ...sx }}>
      <Tabs
        value={value}
        onChange={handleChange}
        sx={{
          flexGrow: 1,
          '& .MuiTabs-indicator': { display: 'none' },
          minHeight: '35px',
          borderBottom: (theme) => hasBorder ? `2px solid ${theme.palette.primary.main}` : undefined,
          '& .MuiTabs-flexContainer': {
            gap: '4px',
          },
        }}
      >
        {tabs.map((tab, index) =>
          isTransparent ?
            <StyledTab
              key={index}
              label={tab.label}
              sx={{ backgroundColor: 'transparent' }}
              disabled={tab.disabled}
              id={`tab-${index}`}
              aria-controls={`tabpanel-${index}`}
            />
            :
            <FormStyledTab
              key={index}
              label={tab.label}
              disabled={tab.disabled}
              id={`tab-${index}`}
              aria-controls={`tabpanel-${index}`}
            />
        )}
      </Tabs>
      {tabs.map((tab, index) => (
        <CustomTabPanel key={index} value={value} index={index}>
          {showNavigationButtons ? (
            <Box
              sx={{
                display: 'flex',
                justifyContent: 'space-between',
                mb: 2,
                gap: 2,
              }}
            >
              <ColumnButton
                variant="contained"
                startIcon={<FontAwesome icon={faArrowLeft} width={14} />}
                onClick={handlePrevious}
                sx={{ visibility: value === 0 ? 'hidden' : 'visible', minWidth: 100, ...lightColor }}
              >
                Précédent
              </ColumnButton>
              <AddButton
                variant="contained"
                endIcon={<FontAwesome icon={faArrowRight} width={14} />}
                onClick={handleNext}
                sx={{ visibility: value === tabs.length - 1 ? 'hidden' : 'visible', minWidth: 100 }}
              >
                Suivant
              </AddButton>
            </Box>
          ) : <></>}
          {tab.content}
        </CustomTabPanel>
      ))}
    </Box>
  );
}
