"use client"

import Checkbox from "@mui/material/Checkbox"
import TableCell from "@mui/material/TableCell"
import SvgColor from "@/shared/components/svg-color"
import type { TableColumn } from "@/shared/types/common"
import type { IEmailAutomation } from "@/shared/types/email-automation"
import { ContratStyledRow } from "@/shared/sections/contract/styles"
import { formatDateTime } from "@/utils/format-time"
import { Chip, IconButton, Tooltip, Box } from "@mui/material"
import TableActionColumn from "@/shared/components/table/table-action-column"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

type Props = {
  handleEdit: (automation: IEmailAutomation) => void
  handleDetails: (automation: IEmailAutomation) => void
  handleArchive: (automation: IEmailAutomation) => void
  selected: boolean
  row: IEmailAutomation
  dense: boolean
  onSelectRow: VoidFunction
  columns: TableColumn[]
}

const TOOLTIP_TITLES = {
  duplicate: "Copier règle",
  active: "Actif",
  inactive: "Inactif",
}

const getTriggerColor = (type: "FREQUENCE" | "EVENEMENT") => {
  switch (type) {
    case "FREQUENCE":
      return {
        color: "primary",
        backgroundColor: "primary.lighter",
        borderColor: "primary.light",
      }
    case "EVENEMENT":
      return {
        color: "secondary",
        backgroundColor: "secondary.lighter",
        borderColor: "secondary.light",
      }
    default:
      return {
        color: "default",
        backgroundColor: "grey.200",
        borderColor: "grey.300",
      }
  }
}

export default function EmailAutomationTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleArchive,
  selected,
  onSelectRow,
  columns,
}: Props) {
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "dernierEnvoi":
        return value ? formatDateTime(value) : "-"
      case "declencheur":
        const triggerColors = getTriggerColor(row.declencheur.type)
        let label = ""

        if (row.declencheur.type === "FREQUENCE" && row.declencheur.frequence) {
          label = row.declencheur.frequence.replace(/_/g, " ")
        } else if (row.declencheur.type === "EVENEMENT" && row.declencheur.evenement) {
          label = row.declencheur.evenement.replace(/_/g, " ")
        }

        return (
          <Chip
            label={label}
            size="small"
            variant="outlined"
            sx={{
              '&.MuiChip-root': {
                borderRadius: '8px',
                padding: '9px 0',
                height: '100%',
                width: '150px',
                },
              color: `${triggerColors.color}.main`,
              backgroundColor: triggerColors.backgroundColor,
              borderColor: triggerColors.borderColor,
              fontWeight: "medium",
            }}
          />
        )
      default:
        return value
    }
  }

  return (
    <ContratStyledRow hover style={{ cursor: "pointer" }}>
      {columns
        .filter((column) => column.id !== "isArchive")
        .map((column: any,index:number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: "nowrap", py: dense ? 0 : 2,
              ...(index === 0 && {
                position: 'relative',
              })
            }}
          >
            <ConditionalComponent isValid={index === 0}>  
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{
                  position: 'absolute',
                  top: 0,
                  left: -4,
                  height: '100%',
                  color: row.declencheur.type === "FREQUENCE" ? "primary.main" : "secondary.main",
                }}
              />
            </ConditionalComponent>
            
            {renderCellContent(column.id, row[column.id as keyof IEmailAutomation])}
          </TableCell>
        ))}

<TableCell
        sx={{ px: 1, whiteSpace: 'nowrap' }}
      >
        <Box sx={{ display: "flex", gap: 1 }}>
        <TableActionColumn
            handleEdit={() => handleEdit(row)}
            onToggle={() => handleArchive(row)}
            isActive={row.isArchive}
            tooltipTitles={TOOLTIP_TITLES}
          />
        </Box>
      </TableCell>
    </ContratStyledRow>
  )
}
