"use client"

import type React from "react"
import { useState } from "react"
import {
  Box,
  Card,
  Grid,
  Typography,
  FormControlLabel,
  RadioGroup,
  Radio,
  Divider,
  Switch,
  Button,
  Stack,
  FormGroup,
  Checkbox,
  Alert,
} from "@mui/material"
import { Save } from "@mui/icons-material"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { ScrollBox } from "@/shared/theme/css"

// Remplacer LoadingButton de @mui/lab par un bouton standard avec un état de chargement
function LoadingButton({
  loading,
  children,
  ...props
}: { loading: boolean; children: React.ReactNode } & React.ComponentProps<typeof Button>) {
  return (
    <Button disabled={loading} {...props}>
      {loading ? "Chargement..." : children}
    </Button>
  )
}

export default function NotificationSettings() {
  const [loading, setLoading] = useState(false)
  const [saved, setSaved] = useState(false)
  const [settings, setSettings] = useState({
    frequency: "realtime",
    emailNotifications: true,
    pushNotifications: true,
    smsNotifications: false,
    notifyOnSystemUpdates: true,
    notifyOnProfileChanges: true,
    notifyOnActivityUpdates: true,
    notifyOnAlerts: true,
    notifyOnInfo: false,
  })

  const handleFrequencyChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setSettings({
      ...settings,
      frequency: (event.target as HTMLInputElement).value,
    })
    setSaved(false)
  }

  const handleSwitchChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setSettings({
      ...settings,
      [event.target.name]: event.target.checked,
    })
    setSaved(false)
  }

  const handleSave = async () => {
    setLoading(false)
    setSaved(true)
  }

  const handleReset = () => {
    setSettings({
      frequency: "realtime",
      emailNotifications: true,
      pushNotifications: true,
      smsNotifications: false,
      notifyOnSystemUpdates: true,
      notifyOnProfileChanges: true,
      notifyOnActivityUpdates: true,
      notifyOnAlerts: true,
      notifyOnInfo: false,
    })
    setSaved(false)
  }

  return (
    <ScrollBox >
      <ConditionalComponent isValid={saved}>
      <Alert severity="success" sx={{ mb: 2 }}>
          Paramètres de notification enregistrés avec succès!
        </Alert>
      </ConditionalComponent>

      <Card sx={{ px: 3,pb: 2 }}>

          <Typography sx={{color: "primary.main",py: 2}} variant="h4" component="h1" fontWeight="bold">
            Paramètres des Notifications
          </Typography>
        <Grid container spacing={3}>
          <Grid item xs={12}>
            <Typography variant="subtitle1" gutterBottom>
              Fréquence des Notifications
            </Typography>
            <RadioGroup name="frequency" value={settings.frequency} onChange={handleFrequencyChange}>
              <FormControlLabel value="realtime" control={<Radio />} label="En temps réel" />
              <Typography variant="caption" color="text.secondary" sx={{ ml: 4, mb: 1 }}>
                Recevez les notifications dès qu&apos;elles sont envoyées
              </Typography>

              <FormControlLabel value="daily" control={<Radio />} label="Quotidiennement" />
              <Typography variant="caption" color="text.secondary" sx={{ ml: 4, mb: 1 }}>
                Recevez un résumé quotidien de toutes vos notifications
              </Typography>

              <FormControlLabel value="weekly" control={<Radio />} label="Hebdomadairement" />
              <Typography variant="caption" color="text.secondary" sx={{ ml: 4 }}>
                Recevez un résumé hebdomadaire de toutes vos notifications
              </Typography>
            </RadioGroup>
          </Grid>

          <Grid item xs={12}>
            <Divider sx={{ my: 2 }} />
            <Typography variant="subtitle1" gutterBottom>
              Canaux de Notification
            </Typography>
            <FormGroup>
              <FormControlLabel
                control={
                  <Switch
                    checked={settings.emailNotifications}
                    onChange={handleSwitchChange}
                    name="emailNotifications"
                  />
                }
                label="Notifications par email"
              />
              <FormControlLabel
                control={
                  <Switch checked={settings.pushNotifications} onChange={handleSwitchChange} name="pushNotifications" />
                }
                label="Notifications push"
              />
              <FormControlLabel
                control={
                  <Switch checked={settings.smsNotifications} onChange={handleSwitchChange} name="smsNotifications" />
                }
                label="Notifications par SMS"
              />
            </FormGroup>
          </Grid>

          <Grid item xs={12}>
            <Divider sx={{ my: 2 }} />
            <Typography variant="subtitle1" gutterBottom>
              Types de Notifications
            </Typography>
            <FormGroup>
              <FormControlLabel
                control={
                  <Checkbox
                    checked={settings.notifyOnSystemUpdates}
                    onChange={handleSwitchChange}
                    name="notifyOnSystemUpdates"
                  />
                }
                label="Mises à jour système"
              />
              <FormControlLabel
                control={
                  <Checkbox
                    checked={settings.notifyOnProfileChanges}
                    onChange={handleSwitchChange}
                    name="notifyOnProfileChanges"
                  />
                }
                label="Changements de profil"
              />
              <FormControlLabel
                control={
                  <Checkbox
                    checked={settings.notifyOnActivityUpdates}
                    onChange={handleSwitchChange}
                    name="notifyOnActivityUpdates"
                  />
                }
                label="Mises à jour d'activité"
              />
              <FormControlLabel
                control={
                  <Checkbox checked={settings.notifyOnAlerts} onChange={handleSwitchChange} name="notifyOnAlerts" />
                }
                label="Alertes"
              />
              <FormControlLabel
                control={<Checkbox checked={settings.notifyOnInfo} onChange={handleSwitchChange} name="notifyOnInfo" />}
                label="Informations générales"
              />
            </FormGroup>
          </Grid>
        </Grid>

        <Stack direction="row" justifyContent="flex-end" spacing={2} sx={{ mt: 3 }}>
          <Button variant="outlined" color="inherit" onClick={handleReset}>
            Réinitialiser
          </Button>
          <LoadingButton variant="contained" startIcon={<Save />} loading={loading} onClick={handleSave}>
            Enregistrer les paramètres
          </LoadingButton>
        </Stack>
      </Card>
    </ScrollBox>
  )
}
