import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { ContratStyledRow, StatusChip } from '@/shared/sections/contract/styles';
import { useTheme } from '@mui/material/styles';
import { IRouteTracking } from '@/shared/types/suivi-trajet';
import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { getLabel } from '@/shared/_mock';
import { _STATUT_ROUTE_OPTIONS } from '@/shared/_mock/_suiviTrajet';
import { _TRAJET_TYPE } from '@/shared/_mock/_trajets';
import { Box, IconButton } from '@mui/material';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import FontAwesome from '@/shared/components/fontawesome';
import { ActionsIconButton } from '@/shared/theme/css';
import { faFolder, faFolderOpen, faMap, faMapMarkedAlt } from '@fortawesome/free-solid-svg-icons';
import { useBoolean } from '@/hooks';
import SvgColor from '@/shared/components/svg-color';
import StatusMenu from '@/shared/components/table/StatusMenu';
import { getStatusColor } from '@/shared/sections/rh/styles';
import { AntSwitch } from '@/shared/components/table/styles';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEdit: (suiviTrajet: IRouteTracking) => void;
  handleDetails: (suiviTrajet: IRouteTracking) => void;
  handleCopy: (suiviTrajet: IRouteTracking) => void;
  selected: boolean;
  row: IRouteTracking;
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IRouteTracking, newStatus: string, field: keyof IRouteTracking) => void;
};

const TOOLTIP_TITLES = {
  duplicate: 'Copier trajet',
  active: 'Trajet actif',
  inActive: 'Trajet inactif',
};

export default function SuiviTrajetTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  selected,
  onSelectRow,
  columns,
  onStatusChange,
}: Props) {
  const active = useBoolean(row.isActive);
  const theme = useTheme();
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'dateDebut':
        return formatDate(value, dateFormat.isoDate);
      case 'statut':
        return (
          <StatusMenu
            value={value}
            options={_STATUT_ROUTE_OPTIONS}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'statut')}
            chipComponent={StatusChip}
          />
        );
      case 'typeTrajet':
        return (
          <StatusMenu
            value={value}
            options={_TRAJET_TYPE}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'typeTrajet')}
            chipComponent={StatusChip}
            sx={{ width: '170px' }}
          />
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>
      {columns
        .slice(0, -1)
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align || 'left'}
            sx={{
              whiteSpace: 'nowrap',
              ...(index === 0 && {
                position: 'relative',
              }),
              py: column.id === 'statut' || column.id === 'typeTrajet' ? 0 : dense ? 1 : 2,
            }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{
                  position: 'absolute',
                  top: 0,
                  left: -4,
                  height: '100%',
                  color: getStatusColor(row.intitule),
                }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IRouteTracking])}
          </TableCell>
        ))}

      <TableCell sx={{ px: 1, whiteSpace: 'nowrap', py: 0 }}>
        <Box sx={{ display: 'flex', justifyContent: 'left', alignItems: 'center', gap: '8px' }}>
          <CustomTooltip title="Consulter trajet" arrow>
            <IconButton onClick={() => handleEdit(row)} size="small" sx={ActionsIconButton}>
              <FontAwesome icon={faMapMarkedAlt} width={16} />
            </IconButton>
          </CustomTooltip>

          <CustomTooltip
            title={active.value ? TOOLTIP_TITLES.active : TOOLTIP_TITLES.inActive}
            arrow
          >
            <AntSwitch theme={theme} checked={row.isActive} size="small" />
          </CustomTooltip>
        </Box>
      </TableCell>
    </ContratStyledRow>
  );
}
