import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';
import SvgColor from '@/shared/components/svg-color';
import { TableColumn } from '@/shared/types/common';
import TableActionColumn from '@/shared/components/table/table-action-column';
import { ContratStyledRow, getStatusColor } from '@/shared/sections/contract/styles';
import { alpha, IconButton } from '@mui/material';
import { ActionsIconButton } from '@/shared/theme/css';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { _REGIME_OPTIONS } from '@/shared/_mock/_etablissement';
import { useTheme } from '@mui/material/styles';
import { EstablishmentSector, establishmentSectorLabelsFr, EstablishmentTableItem, EstablishmentType, establishmentTypeLabelsFr } from '@/shared/types/establishment';
import { useEstablishmentStore } from '@/shared/api/stores/admin-service/establishmentStore';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEdit: (etablissement: EstablishmentTableItem) => void;
  handleDetails: (etablissement: EstablishmentTableItem) => void;
  handleCopy: (etablissement: EstablishmentTableItem) => void;
  selected: boolean;
  selectedHoraire: boolean;
  row: EstablishmentTableItem;
  handleHoraireClick: (etablissement: EstablishmentTableItem) => void
  dense: boolean;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
};

const TOOLTIP_TITLES = {
  duplicate: 'Copier établissement',
  active: 'Établissement Actif',
  inactive: 'Établissement Inactif',
};

export default function EstablishmentTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  handleHoraireClick,
  selected,
  selectedHoraire,
  onSelectRow,
  columns,
}: Props) {
  const theme = useTheme()

  const { toogleActivation } = useEstablishmentStore();

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'horaire':
        return (
          <CustomTooltip title="Horaires de l'établissement" arrow>
            <IconButton
              sx={{ py: 0, px: 0.5, ...ActionsIconButton }}
              onClick={(e) => {
                e.stopPropagation()
                handleHoraireClick(row)
              }}
            >
              <SvgColor
                src="/assets/icons/ic_date.svg"
                sx={{
                  width: 14,
                }}
              />
            </IconButton>
          </CustomTooltip>
        );
      case "establishmentSector":
        return establishmentSectorLabelsFr[value as EstablishmentSector]
      case "establishmentType":
        return establishmentTypeLabelsFr[value as EstablishmentType]
      case 'address':
        return (
          row.address.street + ' ' + row.address.zipCode + ' ' + row.address.city + ' ' + row.address.country
        );
      default:
        return value;
    }
  };
  return (
    <ContratStyledRow hover style={{
      backgroundColor: selectedHoraire ? alpha(theme.palette.primary.main, 0.1) : undefined,
      cursor: 'pointer'
    }}>
      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: dense ? 0 : 2, ...(index === 0 && {
              position: 'relative',
            }), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: 'absolute', top: 0, left: -4, height: '100%', color: getStatusColor(row.establishmentType),
              }}
            />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof EstablishmentTableItem])}
          </TableCell>
        ))}
      <TableCell sx={{ px: 1, whiteSpace: 'nowrap', py: 0 }}>
        <TableActionColumn
          handleEdit={() => handleEdit(row)}
          onDuplicate={() => handleCopy(row)}
          onToggle={(checked) => toogleActivation(row.id, checked)}
          isActive={row.active}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}