"use client"

import React, { useEffect, useState } from "react"
import { Box, Stack, Tab, Typography } from "@mui/material"
import { HEADER } from "@/layouts/config-layout"
import TableControlBar from "@/shared/components/table/table-control-bar"
import { TableType } from "@/shared/types/common"
import { type IExport, ExportTab } from "@/shared/types/export"
import TableManager from "@/shared/components/table/table-manager"
import ExportTableRow from "./export-table-row"
import { StyledFilterTabs } from "@/shared/theme/css"
import { useExportTable } from "../hooks/use-export-table"

export default function ExportListView() {
  const [activeMainTab, setActiveMainTab] = useState<ExportTab>(ExportTab.ALL)

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    handleSave,
    handleStatusChange,
    dataToExport,
  } = useExportTable()

  useEffect(() => {
    if (activeMainTab === ExportTab.ALL) {
      handleResetFilters()
    } else {
      handleFilterChange("exportFilesStatus", activeMainTab, "equals")
    }
  }, [activeMainTab, handleFilterChange, handleResetFilters])

  const handleTabChangeWrapper = (event: React.SyntheticEvent, newValue: ExportTab) => {
    setActiveMainTab(newValue)
  }


  const renderRow = (row: IExport) => (
    <ExportTableRow
      key={row.id}
      row={row}
      handleDownloadExport={()=> void 0}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead.slice(0, -1)}
    />
  )

  return (
    <Box
      sx={{
        display: "flex",
        flexDirection: "column",
        height: `calc(100vh - ${HEADER.H_DESKTOP + 8}px)`,
      }}
    >
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.Export}
          activeTab={"list"}
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={dataToExport}
        />
      </Stack>
      <Typography
          variant="h5"
          sx={{
            mb: 2,
            color: 'primary.main',
            fontWeight: (theme) => theme.typography.fontWeightBold,
          }}
        >
          Liste des exports de fichiers
        </Typography>
    
        <StyledFilterTabs value={activeMainTab} onChange={handleTabChangeWrapper} sx={{ mb: 2 }}>
          <Tab label="Tous" value={ExportTab.ALL} />
          <Tab label="Complété" value={ExportTab.COMPLETED} />
          <Tab label="En cours" value={ExportTab.IN_PROGRESS} />
          <Tab label="Échoué" value={ExportTab.FAILED} />
        </StyledFilterTabs>

        <TableManager
          filteredData={filteredData}
          table={table}
          tableHead={tableHead}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderRow}
          isWithoutTabs
        />

    </Box>
  )
}
