'use client';

import React, { useState } from 'react';
import { Box, Stack, Tabs, Typography } from '@mui/material';
import TableControlBar from '@/shared/components/table/table-control-bar';
import { DrawerControlProps, TableType } from '@/shared/types/common';
import { IMileage } from '@/shared/types/flottes/mileage';
import TableManager from '@/shared/components/table/table-manager';
import TreatedMileageTableRow from './treated-mileage-table-row';
import { useBoolean } from '@/hooks';
import { _mileageList, MILEAGE_TABLE_HEAD } from '@/shared/_mock/flottes/_mileage';
import { TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { _consumptionMockList } from '@/shared/_mock/flottes/_employee';
import { StyledTab, tabsStyles } from '@/shared/theme/css';
import TreatedMileAgeDrawer from './treated-mileage-details';

const treatedMileageTableConfig: TableConfig<IMileage> = {
  initialData: _mileageList,
  defaultTableHead: [...MILEAGE_TABLE_HEAD, { id: 'circuit', label: 'Circuit', type: 'text' }],
};

export default function TreatedMileageTable({ openDrawer }: DrawerControlProps) {
  const [selectedRowForDrawer, setSelectedRowForDrawer] = useState<IMileage | null>(null);

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    dataToExport
  } = useTableManager<IMileage>(treatedMileageTableConfig);

  const handleRowDoubleClick = (row: IMileage) => {
    setSelectedRowForDrawer(row);
    openDrawer.onTrue();
  };

  const renderRow = (row: IMileage) => (
    <TreatedMileageTableRow
      key={row.id}
      row={row}
      onRowDoubleClick={() => handleRowDoubleClick(row)}
      selected={table.selected.includes(row.id)}
      onSelectRow={() => table.onSelectRow(row.id)}
      columns={tableHead}
      dense={table.dense}
    />
  );

  return (
    <Box >
      <Typography
        variant="h5"
        sx={{
          color: 'primary.main',
          fontWeight: (theme) => theme.typography.fontWeightBold,
        }}
      >
        Suivie Kilomètrage
      </Typography>
      <Stack
        flexGrow={0}
        direction="row"
        alignItems="center"
        justifyContent="flex-end"
        spacing={{ xs: 0.5, sm: 1 }}
      >
        <TableControlBar
          type={TableType.Mileage}
          activeTab="list"
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead}
          onFilters={handleFilterChange}
          filteredData={dataToExport}
               />
      </Stack>

      <Tabs value={'list'} sx={tabsStyles}>
        <StyledTab value={'list'} label={"Liste des anomalies d'accès Kilométrage traitées"} />
      </Tabs>

      <TableManager
        filteredData={filteredData}
        table={table}
        tableHead={tableHead}
        notFound={notFound}
        filters={filters}
        onFilterChange={handleFilterChange}
        renderRow={renderRow}
        fixedHeight="calc(100vh - 320px)"
      />

      {selectedRowForDrawer ? (
        <TreatedMileAgeDrawer
          isOpen={openDrawer}
          onClose={openDrawer.onFalse}
          rowData={selectedRowForDrawer}
        />
      ) : (
        <></>
      )}
    </Box>
  );
}
