import React from 'react';
import Checkbox from '@mui/material/Checkbox';
import TableCell from '@mui/material/TableCell';

import TableActionColumn from '@/shared/components/table/table-action-column';
import { useTheme } from '@mui/material/styles';
import SvgColor from '@/shared/components/svg-color';
import StatusMenu from '@/shared/components/table/StatusMenu';

import { TableColumn } from '@/shared/types/common';
import { dateFormat, formatDate } from '@/utils/format-time';
import { IVehicle } from '@/shared/types/flottes/vehicule';
import { ContratStyledRow, getStatusColor, StatusChip } from '@/shared/sections/contract/styles';
import { _VEHICLE_OPTION_STATUT } from '@/shared/_mock/flottes/_vehicle';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

type Props = {
  handleEditVehicle: (vehicle: IVehicle) => void;
  handleDetailsVehicle: (vehicle: IVehicle) => void;
  handleCopyVehicle: (vehicle: IVehicle) => void;
  selected: boolean;
  row: IVehicle;
  onSelectRow: VoidFunction;
  columns: TableColumn[];
  onStatusChange: (row: IVehicle, newStatus: string, field: keyof IVehicle) => void;
};

const TOOLTIP_TITLES = {
  edit: 'Modifier véhicule',
  duplicate: 'Copier véhicule',
  active: 'Véhicule actif',
  inactive: 'Véhicule inactif',
};

export default function VehicleTableRow({
  row,
  handleEditVehicle,
  handleDetailsVehicle,
  handleCopyVehicle,
  selected,
  onSelectRow,
  columns,
  onStatusChange
}: Props) {

  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case 'dateImmatriculation':
        return formatDate(value, dateFormat.isoDate);
      case 'providerType':
        return value.intitule;
      case 'statut':
        return (
          <StatusMenu
            value={value}
            options={_VEHICLE_OPTION_STATUT}
            onChange={(newStatus) => onStatusChange(row, newStatus, 'statut' as keyof IVehicle)}
            chipComponent={(props) => <StatusChip status={props.value} {...props} />}
            sx={{width:"180px"}}
          />
        );
      default:
        return value;
    }
  };

  return (
    <ContratStyledRow hover style={{ cursor: 'pointer' }}>

      {columns
        .filter((column) => column.id !== 'isActive')
        .map((column: any, index: number) => (
          <TableCell
            onDoubleClick={() => handleDetailsVehicle(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: 'nowrap', py: column.id === 'status' ? 0 : 'inherit', ...(index === 0 && {position: "relative"}), }}
          >
            <ConditionalComponent isValid={index === 0}>
            <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{ position: "absolute", top: 0, left: -4, height: "100%", color: getStatusColor(row.modele) }}
              />
            </ConditionalComponent>
            {renderCellContent(column.id, row[column.id as keyof IVehicle])}
          </TableCell>
        ))}

      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap' }}>
        <TableActionColumn
          handleEdit={() => handleEditVehicle(row)}
          onDuplicate={() => handleCopyVehicle(row)}
          onToggle={(checked) => void 0}
          isActive={row.isActive}
          tooltipTitles={TOOLTIP_TITLES}
        />
      </TableCell>
    </ContratStyledRow>
  );
}