'use client';

import React, { useEffect } from 'react';
import { Grid } from '@mui/material';
import { useFormik } from 'formik';
import FormField from '@/shared/components/form/form-field';
import type { FormFieldType, ModeType } from '@/shared/types/common';
import EditExportButtons from '@/shared/components/form/edit-export-buttons';
import ActionButtons from '@/shared/components/form/buttons-action';
import FormContainer from '@/layouts/form/form-container';
import { incidentTypeFormFields } from '../utils/form-fields-incident-type';
import { ITypeIncident } from '@/shared/types/flottes/incident';
import { incidentTypeSchema } from '../utils/incident-type-schema';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';

interface IncidentTypeFormProps {
  incidentType: ITypeIncident;
  mode: ModeType;
  onSave: (updatedIncidentType: ITypeIncident) => void;
  onEdit: (updatedIncidentType: ITypeIncident) => void;
  onClose: () => void;
  setHasUnsavedChanges?: (value: boolean) => void
  updateTabContent?: (tabId: string, newContent: ITypeIncident) => void
  tabId?: string
}

export default function IncidentTypeForm({
  incidentType,
  mode,
  onSave,
  onClose,
  onEdit,
  setHasUnsavedChanges,
  updateTabContent,
  tabId,
}: IncidentTypeFormProps) {
  const isReadOnly = mode === 'view';

  const formik = useFormik<ITypeIncident>({
    initialValues: incidentType,
    validationSchema: incidentTypeSchema,
    onSubmit: (values) => {
      onSave(values);
      if (setHasUnsavedChanges) setHasUnsavedChanges(false)
      onClose();
    },
  });
  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const renderField = (field: FormFieldType<ITypeIncident>) => (
    <FormField
      field={field}
      value={formik.values[field.name as keyof ITypeIncident]}
      onChange={(name, value) => formik.setFieldValue(name, value)}
      error={
        formik.touched[field.name as keyof ITypeIncident]
          ? (formik.errors[field.name as keyof ITypeIncident] as string | undefined)
          : undefined
      }
      onBlur={formik.handleBlur}
      isReadOnly={isReadOnly}
    />
  );
  return (
    <FormContainer titre="Type d'incident">
      <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons onEdit={() => onEdit(formik.values)} tooltipTitle="le type d'incident" />
      </ConditionalComponent>

      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={5}>
          {incidentTypeFormFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}

          <ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
              <ActionButtons
                onSave={() => {
                  formik.validateForm().then((errors) => {
                    formik.submitForm()
                  })
                }}
                onCancel={onClose}
                mode={mode}
              />
            </Grid>
          </ConditionalComponent>
        </Grid>
      </form>
    </FormContainer>
  )
}
