"use client"

import React, { useEffect } from "react"
import { Grid } from "@mui/material"
import { useFormik } from "formik"
import FormField from "@/shared/components/form/form-field"
import type { FormFieldType, ModeType } from "@/shared/types/common"
import EditExportButtons from "@/shared/components/form/edit-export-buttons"
import ActionButtons from "@/shared/components/form/buttons-action"
import type { IVehicleModel } from "@/shared/types/flottes/vehicule"
import { vehicleModelFormFields } from "../utils/form-fields-vehicle-model"
import FormContainer from "@/layouts/form/form-container"
import { vehicleModelSchema } from "../utils/vehicle-model-schema"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

interface VehicleModelFormProps {
  vehicleModel: IVehicleModel
  mode: ModeType
  onSave: (updatedVehicleModel: IVehicleModel) => void
  onEdit: (updatedVehicleModel: IVehicleModel) => void
  onClose: () => void
  setHasUnsavedChanges?: (value: boolean) => void
  updateTabContent?: (tabId: string, newContent: IVehicleModel) => void
  tabId?: string
}

export default function VehicleModelForm({ vehicleModel, mode, onSave, onClose, onEdit, setHasUnsavedChanges, updateTabContent, tabId }: VehicleModelFormProps) {
  const isReadOnly = mode === "view"

  const formik = useFormik<IVehicleModel>({
    initialValues: vehicleModel,
    validationSchema: vehicleModelSchema,
    onSubmit: (values) => {
      onSave(values)
      if (setHasUnsavedChanges) setHasUnsavedChanges(false)
      onClose()
    },
  })
  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const renderField = (field: FormFieldType<IVehicleModel>) => (
    <FormField
      field={field}
      value={formik.values[field.name as keyof IVehicleModel]}
      onChange={(name, value) => formik.setFieldValue(name, value)}
      error={
        formik.touched[field.name as keyof IVehicleModel]
          ? (formik.errors[field.name as keyof IVehicleModel] as string | undefined)
          : undefined
      }
      onBlur={formik.handleBlur}
      isReadOnly={field.disabled || isReadOnly}
    />
  )

  return (
    <FormContainer titre="Marque / Modèle">
      <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons onEdit={() => onEdit(formik.values)} tooltipTitle="le modèle" />
      </ConditionalComponent>

      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={5}>
          {vehicleModelFormFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}

          <ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
              <ActionButtons
                onSave={() => {
                  formik.validateForm().then((errors) => {
                    formik.submitForm()
                  })
                }}
                onCancel={onClose}
                mode={mode}
              />
            </Grid>
          </ConditionalComponent>
        </Grid>
      </form>
    </FormContainer>
  )
}
