"use client"

import React, { useEffect } from "react"
import { Grid } from "@mui/material"
import { useFormik } from "formik"
import FormField from "@/shared/components/form/form-field"
import type { FormFieldType, ModeType } from "@/shared/types/common"
import EditExportButtons from "@/shared/components/form/edit-export-buttons"
import ActionButtons from "@/shared/components/form/buttons-action"
import FormContainer from "@/layouts/form/form-container"
import { reparationTypeFormFields } from "../utils/form-fields-reparation-type"
import { ITypeReparation } from "@/shared/types/flottes/intervention"
import { reparationTypeSchema } from "../utils/reparation-type-schema"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

interface ReparationTypeFormProps {
  reparationType: ITypeReparation
  mode: ModeType
  onSave: (updatedReparationType: ITypeReparation) => void
  onEdit: (updatedReparationType: ITypeReparation) => void
  onClose: (isSaved: boolean) => void
  updateTabContent?: (tabId: string, newContent: ITypeReparation) => void
  tabId?: string
}

export default function ReparationTypeForm({
  reparationType,
  mode,
  onSave,
  onClose,
  onEdit,
  updateTabContent,
  tabId,
}: ReparationTypeFormProps) {
  const isReadOnly = mode === "view"

  const formik = useFormik<ITypeReparation>({
    initialValues: reparationType,
    validationSchema: reparationTypeSchema,
    onSubmit: (values) => {
      onSave(values)
      onClose(true)
    },
  })
  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const renderField = (field: FormFieldType<ITypeReparation>) => (
    <FormField
      field={field}
      value={formik.values[field.name as keyof ITypeReparation]}
      onChange={(name, value) => formik.setFieldValue(name, value)}
      error={
        formik.touched[field.name as keyof ITypeReparation]
          ? (formik.errors[field.name as keyof ITypeReparation] as string | undefined)
          : undefined
      }
      onBlur={formik.handleBlur}
      isReadOnly={isReadOnly}
    />
  )


  return (
    <FormContainer titre="Type de réparation">
      <ConditionalComponent isValid={isReadOnly}>
        <EditExportButtons
          onEdit={() => onEdit(formik.values)}
          tooltipTitle="le type de réparation"
        />
      </ConditionalComponent>
      
      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={5}>
          {reparationTypeFormFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}

          <ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
              <ActionButtons
                onSave={() => {
                  formik.validateForm().then((errors) => {
                    formik.submitForm()
                  })
                }}
                onCancel={() => onClose(false)}
                mode={mode}
              />
            </Grid>
          </ConditionalComponent>
        </Grid>
      </form>
    </FormContainer>
  )
}
