import React from 'react';
import FontAwesome from '@/shared/components/fontawesome';
import { pxToRem } from '@/shared/theme/typography';
import { faCar, faCalendarDays, faTag, faBuilding, faInfoCircle, faIdCard } from '@fortawesome/free-solid-svg-icons';
import { faOrcid } from '@fortawesome/free-brands-svg-icons';
import { Box, Popover, Typography } from '@mui/material';
import { getBgColor, StatusColor } from '../utils/schedule-utils';

interface TimeSlotPopoverProps {
  anchorEl: HTMLElement | null;
  onClose: () => void;
  color: StatusColor;
  data: {
    vehicle: string;
    vehicleName?: string;
    immatriculation?: string;
    vehicleType: string;
    supplier: string;
    period: string;
    status: string;
    notes?: string;
  };
}

const DetailRow: React.FC<{
  icon: any;
  label: string;
  value?: string;
  color: StatusColor;
}> = ({ icon, label, value, color }) => (
  <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, mb: 1 }}>
    <FontAwesome icon={icon} color={color} width={13} />
    <Typography sx={{ fontSize: pxToRem(12), fontWeight: 'bold', color: { color } }}>
      {label}
    </Typography>
    <Typography sx={{ fontSize: pxToRem(12), color: { color } }}>
      {value || 'Non spécifié'}
    </Typography>
  </Box>
);

export function TrajetDetailsPopover({ anchorEl, onClose, color, data }: TimeSlotPopoverProps) {
  const bgColor = getBgColor(color);

  if (!data) {
    return null;
  }

  const details = [
    { icon: faCar, label: 'Nom de Véhicule', value: data.vehicle },
    { icon: faIdCard, label: 'Immatriculation', value: data.immatriculation },
    { icon: faOrcid, label: 'Type de véhicule', value: data.vehicleType },
    { icon: faBuilding, label: 'Fournisseur', value: data.supplier },
    { icon: faCalendarDays, label: 'Période', value: data.period },
    { icon: faTag, label: 'Statut', value: data.status },
    { icon: faInfoCircle, label: 'Observation', value: data.notes, showOnlyIfValue: true }
  ];

  return (
    <Popover
      open={Boolean(anchorEl)}
      anchorEl={anchorEl}
      onClose={onClose}
      disableRestoreFocus
      anchorOrigin={{
        vertical: 'center',
        horizontal: 'center',
      }}
      transformOrigin={{
        vertical: 'top',
        horizontal: 'center',
      }}
      sx={{
        pointerEvents: 'none',
        '& .MuiPopover-paper': {
          p: 1,
          maxWidth: 300,
          backgroundColor: bgColor,
          border: `2px solid ${color}`,
          borderRadius: 2,
          '&::before': {
            content: '""',
            display: 'block',
            position: 'absolute',
            top: '50%',
            left: -10,
            width: 20,
            height: 20,
            backgroundColor: bgColor,
            transform: 'rotate(45deg)',
            borderRadius: 1,
            border: `2px solid ${color}`,
            borderBottom: 'none',
            borderRight: 'none',
            zIndex: 1,
          },
        },
      }}
    >
      <Box sx={{ display: 'flex', flexDirection: 'column', gap: 0 }}>
        {details.map((detail, index) => (
          detail.showOnlyIfValue ? (
            detail.value && (
              <DetailRow
                key={index}
                icon={detail.icon}
                label={detail.label}
                value={detail.value}
                color={color}
              />
            )
          ) : (
            <DetailRow
              key={index}
              icon={detail.icon}
              label={detail.label}
              value={detail.value}
              color={color}
            />
          )
        ))}
      </Box>
    </Popover>
  );
}
