import { addDays, addMinutes, differenceInDays, parseISO, isWithinInterval } from "date-fns"
import type { TimeSlot } from "../types"

export const calculateTimeSlotWidth = (
  startDate: string,
  endDate: string,
  cellWidth: number,
) => {
  try {
    const start = parseISO(startDate)
    const end = parseISO(endDate)
    const duration = differenceInDays(end, start) + 1 
    return duration * cellWidth - 5
  } catch (error) {
    console.error("Error calculating time slot width:", error)
    return cellWidth - 5 
  }
}

export const isTimeSlotInCell = (slot: TimeSlot, cellDate: string) => {
  try {
    const slotStart = parseISO(slot.startDate)
    const slotEnd = parseISO(slot.endDate)
    const currentDate = parseISO(cellDate)
    
    return isWithinInterval(currentDate, { start: slotStart, end: slotEnd })
  } catch (error) {
    console.error("Error checking if time slot is in cell:", error, { slot, cellDate })
    return false
  }
}

// This function checks if a date is the start date of the time slot
export const isTimeSlotStartDate = (slot: TimeSlot, cellDate: string) => {
  try {
    return slot.startDate === cellDate
  } catch (error) {
    console.error("Error checking if time slot starts in cell:", error, { slot, cellDate })
    return false
  }
}

export const groupOverlappingSlots = (slots: TimeSlot[]) => {
  if (slots.length === 0) return []

  const sortedSlots = [...slots].sort((a, b) => {
    return parseISO(a.startDate).getTime() - parseISO(b.startDate).getTime()
  })

  const groups: TimeSlot[][] = []
  let currentGroup: TimeSlot[] = [sortedSlots[0]]
  let currentEnd = parseISO(sortedSlots[0].endDate)

  for (let i = 1; i < sortedSlots.length; i++) {
    const slot = sortedSlots[i]
    const slotStart = parseISO(slot.startDate)
    const slotEnd = parseISO(slot.endDate)

    if (slotStart <= currentEnd) {
      currentGroup.push(slot)
      if (slotEnd > currentEnd) {
        currentEnd = slotEnd
      }
    } else {
      groups.push(currentGroup)
      currentGroup = [slot]
      currentEnd = slotEnd
    }
  }

  groups.push(currentGroup)
  return groups
}

export const findSlotIndexInGroup = (slot: TimeSlot, overlappingGroups: TimeSlot[][]) => {
  for (const group of overlappingGroups) {
    const index = group.indexOf(slot)
    if (index !== -1) return index
  }
  return 0
}

export const getHourIntervals = (selectedDate: Date) => {
  const intervals = []
  let currentTime = new Date(selectedDate)
  currentTime.setHours(0, 0, 0)

  for (let i = 0; i < 24; i++) {
    intervals.push({
      hour: currentTime.getHours().toString().padStart(2, "0"),
      startCell: i * 4,
      endCell: (i + 1) * 4 - 1,
    })
    currentTime = addMinutes(currentTime, 60)
  }
  return intervals
}

export enum StatusColor  {
  ERROR = "#FF3434",
  SUCCESS = "#30BA1D",
  WARNING = "#F9B90B",
  INFO = "#5176FE",
  PRIMARY = "#746CD4",
  GREY = "#919EAB",
}

export const getBgColor = (color: StatusColor) => {
  switch (color) {
    case StatusColor.ERROR:
      return "#FFEBEB";
    case StatusColor.SUCCESS:
      return "#EAF8E8";
    case StatusColor.WARNING:
      return "#FEF8E7";
    case StatusColor.INFO:
      return "#EEF1FF";
    case StatusColor.GREY:
      return "#DFE3E8";
    case StatusColor.PRIMARY:
      return 'primary.lighter';
    default:
      return 'primary.lighter';
  }
};
