import { 
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Grid,
  IconButton,
  Box,
  styled
} from '@mui/material';
import { useState } from 'react';
import { format } from 'date-fns';
import FormField from '@/shared/components/form/form-field';
import { 
  faInfoCircle,
  faCalendar,
  faClock,
  faUser,
  faCircleCheck,
  faComments,
  faCar,
  faXmark
} from '@fortawesome/free-solid-svg-icons';
import FontAwesome from '@/shared/components/fontawesome';
import { InterventionFormField } from '../types/intervention-form';
// Styled components
const StyledDialog = styled(Dialog)(({ theme }) => ({
  '& .MuiDialog-paper': {
    borderRadius: '16px',
    maxWidth: '600px',
    width: '100%'
  }
}));

const StyledDialogTitle = styled(DialogTitle)(({ theme }) => ({
  backgroundColor: theme.palette.primary.main,
  color: theme.palette.common.white,
  padding: theme.spacing(2),
  display: 'flex',
  alignItems: 'center',
  justifyContent: 'space-between'
}));

const StyledDialogContent = styled(DialogContent)(({ theme }) => ({
  padding: theme.spacing(3),
  '& .MuiFormControl-root': {
    '& .MuiOutlinedInput-root': {
      borderRadius: '12px',
      backgroundColor: theme.palette.background.paper,
      '& fieldset': {
        borderColor: theme.palette.primary.lighter,
      },
      '&:hover fieldset': {
        borderColor: theme.palette.primary.main,
      },
    },
  }
}));

const StyledDialogActions = styled(DialogActions)(({ theme }) => ({
  padding: theme.spacing(2, 3),
  '& .MuiButton-root': {
    borderRadius: '8px',
    textTransform: 'uppercase',
    fontWeight: 600,
    padding: theme.spacing(1, 3),
  }
}));

// Constants
const VEHICLES = [
  { immatriculation: 'AA-123-BB', marque: 'Mercedes', modele: 'Sprinter' },
  { immatriculation: 'BB-456-CC', marque: 'Ford', modele: 'Transit' },
  { immatriculation: 'CC-789-DD', marque: 'Renault', modele: 'Master' },
  { immatriculation: 'DD-012-EE', marque: 'Fiat', modele: 'Ducato' },
  { immatriculation: 'EE-345-FF', marque: 'Mercedes', modele: 'Sprinter' },
];

const VEHICLE_OPTIONS = VEHICLES.map(vehicle => ({
  value: vehicle.immatriculation,
  label: `${vehicle.marque} ${vehicle.modele} - ${vehicle.immatriculation}`
}));

const TECHNICIAN_OPTIONS = [
  { value: 'John Doe', label: 'John Doe' },
  { value: 'Jane Smith', label: 'Jane Smith' },
  { value: 'Bob Wilson', label: 'Bob Wilson' },
  { value: 'Alice Brown', label: 'Alice Brown' }
];

const INTERVENTION_TYPES = [
  { value: 'preventive', label: 'Entretien préventif' },
  { value: 'corrective', label: 'Maintenance corrective' }
];

const STATUS_OPTIONS = [
  { value: 'planned', label: 'Planifié' },
  { value: 'in_progress', label: 'En cours' },
  { value: 'completed', label: 'Terminé' }
];

const initialFormState = {
  type: 'preventive',
  vehicle: '',
  date: format(new Date(), 'yyyy-MM-dd'),
  time: '23:59',
  estimatedDuration: 60,
  technician: '',
  status: 'planned',
  interventionDetails: ''
};

interface AddInterventionDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (eventData: any) => void;
}

export default function AddInterventionDialog({ open, onClose, onSubmit }: AddInterventionDialogProps) {
  const [formData, setFormData] = useState(initialFormState);

  const handleChange = (field: string, value: any) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSubmit(formData);
    setFormData(initialFormState);
    onClose();
  };

  return (
    <StyledDialog open={open} onClose={onClose} fullWidth>
      <StyledDialogTitle>
        Planifier une intervention
        <IconButton 
          onClick={onClose}
          sx={{ 
            color: 'common.white',
            '&:hover': { bgcolor: 'primary.dark' }
          }}
        >
          <FontAwesome icon={faXmark} width={20} />
        </IconButton>
      </StyledDialogTitle>

      <form onSubmit={handleSubmit}>
        <StyledDialogContent>
          <Grid container spacing={3}>
            <Grid item xs={12}>
              <FormField
                field={{
                  name: 'type',
                  label: "Type d'intervention",
                  type: 'select',
                  icon: faInfoCircle,
                  options: INTERVENTION_TYPES
                }}
                value={formData.type}
                onChange={(name, value) => handleChange('type', value)}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12}>
              <FormField
                field={{
                  name: 'vehicle',
                  label: 'Véhicule concerné',
                  type: 'select',
                  icon: faCar,
                  options: VEHICLE_OPTIONS
                }}
                value={formData.vehicle}
                onChange={(name, value) => handleChange('vehicle', value)}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12} sm={6}>
              <FormField
                field={{
                  name: 'date',
                  label: 'Date',
                  type: 'date',
                  icon: faCalendar
                }}
                value={formData.date}
                onChange={(name, value) => handleChange('date', value)}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12} sm={6}>
              <FormField
                field={{
                  name: 'time',
                  label: 'Heure',
                  type: 'time',
                  icon: faClock
                }}
                value={formData.time}
                onChange={(name, value) => handleChange('time', value)}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12}>
            <FormField
  field={{
    name: 'estimatedDuration',  
    label: 'Durée estimée (minutes)',
    type: 'number',
    icon: faClock,
    min: 0
  } as InterventionFormField}
                value={formData.estimatedDuration}
                onChange={(name, value) => handleChange('estimatedDuration', parseInt(value))}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12}>
              <FormField
                field={{
                  name: 'technician',
                  label: 'Technicien assigné',
                  type: 'select',
                  icon: faUser,
                  options: TECHNICIAN_OPTIONS
                }}
                value={formData.technician}
                onChange={(name, value) => handleChange('technician', value)}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12}>
              <FormField
                field={{
                  name: 'status',
                  label: 'Statut',
                  type: 'select',
                  icon: faCircleCheck,
                  options: STATUS_OPTIONS
                }}
                value={formData.status}
                onChange={(name, value) => handleChange('status', value)}
                isReadOnly={false}
              />
            </Grid>

            <Grid item xs={12}>
              <FormField
                field={{
                  name: 'interventionDetails',
                  label: "Détails de l'intervention",
                  type: 'textarea',
                  icon: faComments,
                  placeholder: "Ajouter des détails sur l'intervention..."
                }}
                value={formData.interventionDetails}
                onChange={(name, value) => handleChange('interventionDetails', value)}
                isReadOnly={false}
              />
            </Grid>
          </Grid>
        </StyledDialogContent>

        <StyledDialogActions>
          <Button 
            onClick={onClose}
            sx={{ 
              color: 'text.secondary',
              '&:hover': { bgcolor: 'action.hover' }
            }}
          >
            ANNULER
          </Button>
          <Button 
            type="submit" 
            variant="contained" 
            sx={{
              bgcolor: 'primary.main',
              color: 'common.white',
              '&:hover': { bgcolor: 'primary.dark' }
            }}
          >
            PLANIFIER
          </Button>
        </StyledDialogActions>
      </form>
    </StyledDialog>
  );
} 