"use client"

import type React from "react"
import { styled } from "@mui/material/styles"
import { Box, IconButton, Menu, MenuItem } from "@mui/material"
import FontAwesome from "@/shared/components/fontawesome"
import { faWheelchair, faCopy, faCar, faWrench } from "@fortawesome/free-solid-svg-icons"
import SvgColor from "@/shared/components/svg-color"
import { lightColor, greyColor } from "@/shared/components/table/styles"
import type { TimeSlot } from "../types/planning-types"
import { getBgColor } from "../utils/schedule-utils"
import { useMemo, useState } from "react"
import CustomTooltip from "@/shared/components/tooltips/tooltip-custom"
import { iconButtonStyles } from "@/shared/theme/css"

interface TimeSlotBlockProps {
  slot: TimeSlot
  width: number
  top: number
  height: string | number
  handlePopoverOpenT: (event: React.MouseEvent<HTMLElement>, slot: TimeSlot) => void
  handlePopoverCloseT: () => void
  onDoubleClick: (event: React.MouseEvent<HTMLElement>) => void
}

const StyledTimeSlotBlock = styled(Box)(({ theme }) => ({
  position: "absolute",
  top: "0%",
  display: "flex",
  flexDirection: "column",
  justifyContent: "center",
  gap: "3%",
  fontSize: "0.75rem",
  zIndex: 1,
  paddingLeft: 8,
  paddingRight: 36,
}))

const AffectedIndicator = styled(Box)(({ theme }) => ({
  position: "absolute",
  top: 0,
  height: "100%",
  display: "flex",
  pl: 1,
  alignItems: "flex-start",
  justifyContent: "center",
  paddingLeft: "2%",
  zIndex: 0,
  overflow: "hidden",
  flexDirection: "column",
  gap: "3%",
}))

export function VehicleTimeSlotBlock({
  slot,
  width,
  top,
  height,
  handlePopoverOpenT,
  handlePopoverCloseT,
  onDoubleClick,
}: TimeSlotBlockProps) {
  const [contextMenu, setContextMenu] = useState<{
    mouseX: number
    mouseY: number
  } | null>(null)

  const handleDoubleClick = (e: React.MouseEvent<HTMLElement>) => {
    onDoubleClick(e)
  }

  const handleContextMenu = (event: React.MouseEvent) => {
    event.preventDefault()
    setContextMenu(
      contextMenu === null
        ? {
            mouseX: event.clientX,
            mouseY: event.clientY,
          }
        : null,
    )
  }

  const handleCloseContextMenu = () => {
    setContextMenu(null)
  }

  const handleMenuAction = (action: string) => {
    handleCloseContextMenu()
  }

  const bgColor = getBgColor(slot.color)
  const fontSize = useMemo(() => {
    if (width < 150) return 9
    if (width < 200) return 10
    return 11
  }, [width])

  return (
    <>
      <StyledTimeSlotBlock
        onMouseEnter={(e) => handlePopoverOpenT(e, slot)}
        onMouseLeave={handlePopoverCloseT}
        onDoubleClick={handleDoubleClick}
        onContextMenu={handleContextMenu}
        style={{ top: `${top}px`, height }}
        sx={{
          width,
          border: "none",
          pl: 1,
          pr: 2,
          color: slot.color,
          backgroundColor: bgColor,
          borderRadius: 1,
          boxShadow: '0 1px 3px rgba(0, 0, 0, 0.12)',
          transition: 'all 0.2s ease-in-out',
          '&:hover': {
            boxShadow: '0 4px 8px rgba(0, 0, 0, 0.15)',
            filter: 'brightness(0.95)'
          }
        }}
      >
        <Box sx={{ display: "flex", gap: "3%", flexDirection: "column", fontSize: `${fontSize}px` }}>
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <FontAwesome icon={faWrench} width={12} color={slot.color} />
            <span>{slot.title}</span>
          </Box>
          <Box sx={{ display: "flex", alignItems: "center", gap: 1 }}>
            <FontAwesome icon={faCar} width={12} color={slot.color} />
            {`${slot.startHeure}:${slot.startMinute} - ${slot.endHeure}:${slot.endMinute}`}
          </Box>
        </Box>
      </StyledTimeSlotBlock>

      <Menu
        open={contextMenu !== null}
        onClose={handleCloseContextMenu}
        anchorReference="anchorPosition"
        anchorPosition={
          contextMenu !== null
            ? { top: contextMenu.mouseY, left: contextMenu.mouseX }
            : undefined
        }
      >
        <MenuItem onClick={() => handleMenuAction('copy')}>
          <CustomTooltip title={`Copier Trajet`} arrow>
            <IconButton onClick={()=> void 0} size="small" sx={lightColor}>
              <FontAwesome icon={faCopy} width={14} />
            </IconButton>
          </CustomTooltip>
        </MenuItem>
      </Menu>
    </>
  )
}
