import { Box, Drawer, IconButton, Typography, Stack } from '@mui/material';
import InterventionTable, { TABLE_HEAD } from './intervention-table';
import TableControlBar from '@/shared/components/table/table-control-bar';
import { TableType } from '@/shared/types/common';
import { HEADER } from '@/layouts/config-layout';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faArrowLeft } from '@fortawesome/free-solid-svg-icons';
import { useState } from 'react';
import { useTableManager } from '@/hooks/use-table-manager';
import AddInterventionDialog from './add-intervention-dialog';

interface InterventionTableViewProps {
  open: boolean;
  onClose: () => void;
  events: any[];
  onEventClick: (event: any) => void;
  handleResetFilters: () => void;
  handleFilters: (key: string, value: string) => void;
}

export default function InterventionTableView({
  open,
  onClose,
  events,
  onEventClick,
  handleResetFilters,
  handleFilters
}: InterventionTableViewProps) {
  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters: onResetFilters,
    handleResetColumns,
    handleColumnsChange,
  } = useTableManager({
    type: TableType.Intervention,
    initialData: events,
    defaultTableHead: TABLE_HEAD
  });

  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false);

  const handleTabAdd = () => {
    setIsAddDialogOpen(true);
  };

  const handleAddDialogClose = () => {
    setIsAddDialogOpen(false);
  };

  const handleAddIntervention = (eventData: any) => {
  };

  return (
    <>
      <Drawer
        anchor="right"
        open={open}
        onClose={onClose}
        variant="persistent"
        PaperProps={{
          sx: {
            width: '100%',
            maxWidth: '1000px',
            bgcolor: 'background.default',
          },
        }}
      >
        <Box
          sx={{
            display: 'flex',
            flexDirection: 'column',
            height: `calc(100vh - ${HEADER.H_DESKTOP - 20}px)`,
            p: 1,
          }}
        >
          <Box 
            sx={{ 
              display: 'flex', 
              alignItems: 'center', 
              gap: 2, 
              mb: 0.5,
              borderBottom: '1px solid',
              borderColor: 'divider',
              pb: 0.5,
            }}
          >
            <IconButton
              onClick={onClose}
              sx={{
                color: 'primary.main',
                '&:hover': { bgcolor: 'primary.lighter' }
              }}
            >
              <FontAwesomeIcon icon={faArrowLeft} width={16} />
            </IconButton>
            <Typography variant="h6" sx={{ color: 'primary.main', fontWeight: 600 }}>
              Liste des interventions
            </Typography>
          </Box>

          <Stack
            direction="row"
            alignItems="center"
            justifyContent="flex-end"
            spacing={{ xs: 0.5, sm: 1 }}
            sx={{ mb: 0.5 }}
          >
            <TableControlBar
              type={TableType.Intervention}
              activeTab="list"
              handleTabAdd={handleTabAdd}
              onResetFilters={onResetFilters}
              onResetColumns={handleResetColumns}
              onColumnsChange={handleColumnsChange}
              initialColumns={TABLE_HEAD.slice(0, -1)}
              onFilters={handleFilterChange}
              filteredData={filteredData}
              withoutColumnSelector={true}
              buttonAddTitle="Nouvelle intervention"
            />
          </Stack>

          <Box sx={{ 
            flexGrow: 1, 
            overflow: 'auto',
            minHeight: '750px',
          }}>
            <InterventionTable 
              filteredData={filteredData}
              table={table}
              tableHead={tableHead}
              notFound={notFound}
              filters={filters}
              onFilterChange={handleFilterChange}
              handleEdit={onEventClick}
              handleView={onEventClick}
            />
          </Box>
        </Box>
      </Drawer>

      <AddInterventionDialog
        open={isAddDialogOpen}
        onClose={handleAddDialogClose}
        onSubmit={handleAddIntervention}
      />
    </>
  );
} 