import React from 'react';
import { TableColumn } from '@/shared/types/common';
import TableManager from '@/shared/components/table/table-manager';
import { PlanningEvent } from '@/shared/_mock/_planningEvents';
import { DrawerStyledRow } from '@/shared/sections/contract/styles';
import { TableCell, Checkbox, Box, TableRow, styled } from '@mui/material';
import SvgColor from '@/shared/components/svg-color';
import { getStatusColor } from '@/shared/sections/contract/styles';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { IconButton } from '@mui/material';
import FontAwesome from '@/shared/components/fontawesome';
import { faPenToSquare } from '@fortawesome/free-solid-svg-icons';
import { ActionsIconButton } from '@/shared/theme/css';
import { format } from 'date-fns';

const StyledTableRow = styled(TableRow)(({ theme }) => ({
  '&:hover': {
    backgroundColor: theme.palette.primary.lighter,
    cursor: 'pointer',
  },
}));

export const TABLE_HEAD = [
  { 
    id: 'vehicle', 
    label: 'Véhicule', 
    align: 'left' as const, 
    type: 'text',
    searchable: true 
  },
  { 
    id: 'type', 
    label: "Type d'intervention", 
    align: 'left' as const, 
    type: 'text',
    searchable: true,
    options: [
      { value: 'preventive', label: 'Entretien préventif' },
      { value: 'corrective', label: 'Maintenance corrective' }
    ]
  },
  { 
    id: 'date', 
    label: 'Date prévue', 
    align: 'left' as const, 
    type: 'date',
    searchable: true 
  },
  { 
    id: 'status', 
    label: 'Statut', 
    align: 'left' as const, 
    type: 'text',
    searchable: true,
    options: [
      { value: 'planned', label: 'Planifié' },
      { value: 'in_progress', label: 'En cours' },
      { value: 'completed', label: 'Terminé' }
    ]
  },
  { 
    id: 'technician', 
    label: 'Technicien', 
    align: 'left' as const, 
    type: 'text',
    searchable: true 
  },
  { 
    id: 'actions', 
    label: 'Actions', 
    align: 'right' as const 
  }
];

interface InterventionTableProps {
  filteredData: PlanningEvent[];
  table: ReturnType<typeof import('@/shared/components/table/use-tablets').default>;
  tableHead: TableColumn[];
  notFound: boolean;
  filters: Record<string, { value: string; condition: string }>;
  onFilterChange: (key: string, value: string, condition: string) => void;
  handleEdit: (event: PlanningEvent) => void;
  handleView: (event: PlanningEvent) => void;
}

const tableStyles = {
  root: {
    height: '100%',
    position: 'relative',
    '& .MuiTableContainer-root': {
      backgroundColor: 'white',
      borderRadius: 2,
      height: '100%',
      maxHeight: 'none',
      overflow: 'auto',
    },
    '& .MuiTable-root': {
      borderCollapse: 'separate',
      borderSpacing: '0 8px',
      tableLayout: 'auto',
    },
    '& .MuiTableHead-root': {
      position: 'sticky',
      top: 0,
      zIndex: 2,
      '& .MuiTableCell-root': {
        backgroundColor: 'primary.lighter',
        color: 'text.primary',
        fontWeight: 500,
        fontSize: '0.875rem',
        borderBottom: 'none',
        padding: '12px 8px',
      }
    },
    '& .MuiTableBody-root': {
      '& .MuiTableRow-root': {
        backgroundColor: '#FFFFFF',
        '&:hover': {
          backgroundColor: 'primary.lighter',
        }
      },
      '& .MuiTableCell-root': {
        padding: '12px 8px',
        borderBottom: 'none',
      }
    },
    '& .MuiTableCell-stickyHeader': {
      backgroundColor: 'primary.lighter',
    },
    '& .MuiTablePagination-root': {
      position: 'sticky',
      bottom: 0,
      backgroundColor: 'white',
      zIndex: 2,
      borderTop: '1px solid',
      borderColor: 'divider',
    }
  }
};

export default function InterventionTable({
  filteredData,
  table,
  tableHead,
  notFound,
  filters,
  onFilterChange,
  handleEdit,
  handleView,
}: InterventionTableProps) {
  const renderRow = (row: PlanningEvent) => (
    <StyledTableRow key={row.id}>

      <TableCell sx={{
        position: 'relative',
      }}>
        <SvgColor
          src="/assets/icons/ic_border.svg"
          sx={{
            position: 'absolute',
            top: 0,
            left: -4,
            height: '100%',
            color: getStatusColor(row.status),
            display: 'none',
          }}
        />
        {`${row.vehicle.immatriculation} - ${row.vehicle.marque} ${row.vehicle.modele}`}
      </TableCell>
      <TableCell>{row.type === 'preventive' ? 'Entretien préventif' : 'Maintenance corrective'}</TableCell>
      <TableCell>{format(new Date(row.start), 'dd/MM/yyyy HH:mm')}</TableCell>
      <TableCell>{getStatusLabel(row.status)}</TableCell>
      <TableCell>{row.technician}</TableCell>
      <TableCell align="right" sx={{ px: 1, whiteSpace: 'nowrap', py: 0 }}>
        <CustomTooltip title="Voir détails" arrow>
          <IconButton onClick={() => handleView(row)} size="small" sx={ActionsIconButton}>
            <FontAwesome icon={faPenToSquare} width={16} />
          </IconButton>
        </CustomTooltip>
      </TableCell>
    </StyledTableRow>
  );

  return (
    <Box sx={tableStyles.root}>
      <TableManager
        filteredData={filteredData}
        table={table}
        tableHead={tableHead}
        notFound={notFound}
        filters={filters}
        onFilterChange={onFilterChange}
        renderRow={renderRow}
        fixedHeight="750px"
      />
    </Box>
  );
}

function getStatusLabel(status: string) {
  switch(status) {
    case 'planned': return 'Planifié';
    case 'in_progress': return 'En cours';
    case 'completed': return 'Terminé';
    default: return status;
  }
} 