import React from 'react';
import {
  Box,
  Typography,
  Grid,
  TextField,
  FormControl,
  FormControlLabel,
  FormLabel,
  RadioGroup,
  Radio,
  Checkbox,
  Divider,
  Paper,
  Stack,
  Button,
  FormGroup,
  Accordion,
  AccordionSummary,
  AccordionDetails,
  Switch,
} from '@mui/material';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { 
  faUniversalAccess, 
  faEye, 
  faFileExport, 
  faPalette, 
  faCog,
  faInfoCircle,
  faGlobe,
} from '@fortawesome/free-solid-svg-icons';
import ExpandMoreIcon from '@mui/icons-material/ExpandMore';
import { IFormulaire } from '@/shared/types/formulaire';
import { StyledTextField } from '@/shared/theme/css';

interface FieldSettingsProps {
  formulaire: IFormulaire;
  onChange: (settings: Partial<IFormulaire>) => void;
}

export default function FieldSettings({ formulaire, onChange }: FieldSettingsProps) {
  const handleTextChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    
    onChange({
      [name]: value,
    });
  };

  const handleLayoutChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    onChange({
      layout: e.target.value as 'vertical' | 'horizontal',
    });
  };

  const handleThemeChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    onChange({
      theme: e.target.value as 'light' | 'dark',
    });
  };

  const handleAccessibilityChange = (name: string) => (e: React.ChangeEvent<HTMLInputElement>) => {
    onChange({
      accessibilityOptions: {
        ...formulaire.accessibilityOptions,
        [name]: e.target.checked,
      },
    });
  };

  const handleExportOptionChange = (name: string) => (e: React.ChangeEvent<HTMLInputElement>) => {
    onChange({
      exportOptions: {
        ...formulaire.exportOptions,
        [name]: e.target.checked,
      },
    });
  };

  return (
    <Box sx={{ 
      height: '100%', 
      display: 'flex', 
      flexDirection: 'column',
      overflow: 'hidden',
    }}>
      <Typography color={'primary.main'} variant="h6" gutterBottom>
        Paramètres du formulaire
      </Typography>
      
      <Box sx={{ 
        flex: 1, 
        overflow: 'auto', 
        overflowY: 'scroll',
        height: '100%',
        maxHeight: 'calc(100vh - 120px)',
        pr: 2,
        pb: 2,
        '&::-webkit-scrollbar': {
          width: '8px',
        },
        '&::-webkit-scrollbar-track': {
          backgroundColor: 'rgba(0,0,0,0.05)',
          borderRadius: '4px',
        },
        '&::-webkit-scrollbar-thumb': {
          backgroundColor: 'rgba(0,0,0,0.2)',
          borderRadius: '4px',
          '&:hover': {
            backgroundColor: 'rgba(0,0,0,0.3)',
          }
        }
      }}>
        <Grid container spacing={3}>
          {/* General Settings */}
          <Grid item xs={12}>
            <Accordion defaultExpanded>
              <AccordionSummary expandIcon={<ExpandMoreIcon />}>
                <Typography sx={{ display: 'flex', alignItems: 'center' }}>
                  <FontAwesomeIcon icon={faCog} style={{ marginRight: '8px' }} /> 
                  Paramètres généraux
                </Typography>
              </AccordionSummary>
              <AccordionDetails>
                <Grid container spacing={2}>
                  <Grid item xs={12}>
                    <StyledTextField
                      fullWidth
                      label="Titre du formulaire"
                      name="titre"
                      value={formulaire.title || ''}
                      onChange={handleTextChange}
                      required
                    />
                  </Grid>
                  <Grid item xs={12}>
                    <StyledTextField
                      fullWidth
                      label="Description"
                      name="description"
                      value={formulaire.description || ''}
                      onChange={handleTextChange}
                      multiline
                      rows={3}
                    />
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <StyledTextField
                      fullWidth
                      label="Catégorie"
                      name="categorie"
                      value={formulaire.category || ''}
                      onChange={handleTextChange}
                    />
                  </Grid>
                  <Grid item xs={12} md={6}>
                    <StyledTextField
                      fullWidth
                      label="Département"
                      name="departement"
                      value={formulaire.department || ''}
                      onChange={handleTextChange}
                    />
                  </Grid>
                </Grid>
              </AccordionDetails>
            </Accordion>
          </Grid>

          {/* Layout & Appearance */}
          <Grid item xs={12}>
            <Accordion>
              <AccordionSummary expandIcon={<ExpandMoreIcon />}>
                <Typography sx={{ display: 'flex', alignItems: 'center' }}>
                  <FontAwesomeIcon icon={faPalette} style={{ marginRight: '8px' }} /> 
                  Mise en page et apparence
                </Typography>
              </AccordionSummary>
              <AccordionDetails>
                <Grid container spacing={2}>
                  <Grid item xs={12} md={6}>
                    <FormControl component="fieldset">
                      <FormLabel sx={{color :'primary.main'}}>Disposition</FormLabel>
                      <RadioGroup
                        row
                        name="layout"
                        value={formulaire.layout || 'vertical'}
                        onChange={handleLayoutChange}
                      >
                        <FormControlLabel value="vertical" control={<Radio />} label="Vertical" />
                        <FormControlLabel value="horizontal" control={<Radio />} label="Horizontal" />
                      </RadioGroup>
                    </FormControl>
                  </Grid>

                  <Grid item xs={12} md={6}>
                    <FormControl  component="fieldset">
                      <FormLabel sx={{color:'primary.main'}}>Thème</FormLabel>
                      <RadioGroup
                        row
                        name="theme"
                        value={formulaire.theme || 'light'}
                        onChange={handleThemeChange}
                      >
                        <FormControlLabel value="light" control={<Radio />} label="Clair" />
                        <FormControlLabel value="dark" control={<Radio />} label="Sombre" />
                      </RadioGroup>
                    </FormControl>
                  </Grid>
                </Grid>
              </AccordionDetails>
            </Accordion>
          </Grid>

          {/* Accessibility Options */}
          <Grid item xs={12}>
            <Accordion>
              <AccordionSummary expandIcon={<ExpandMoreIcon />}>
                <Typography sx={{ display: 'flex', alignItems: 'center' }}>
                  <FontAwesomeIcon icon={faUniversalAccess} style={{ marginRight: '8px' }} /> 
                  Options d&apos;accessibilité
                </Typography>
              </AccordionSummary>
              <AccordionDetails>
                <FormGroup>
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.accessibilityOptions?.highContrast || false}
                        onChange={handleAccessibilityChange('highContrast')}
                      />
                    }
                    label="Contraste élevé"
                  />
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.accessibilityOptions?.largeText || false}
                        onChange={handleAccessibilityChange('largeText')}
                      />
                    }
                    label="Texte agrandi"
                  />
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.accessibilityOptions?.screenReaderOptimized || false}
                        onChange={handleAccessibilityChange('screenReaderOptimized')}
                      />
                    }
                    label="Optimisé pour les lecteurs d'écran"
                  />
                </FormGroup>
              </AccordionDetails>
            </Accordion>
          </Grid>

          {/* Export Options */}
          <Grid item xs={12}>
            <Accordion>
              <AccordionSummary expandIcon={<ExpandMoreIcon />}>
                <Typography sx={{ display: 'flex', alignItems: 'center' }}>
                  <FontAwesomeIcon icon={faFileExport} style={{ marginRight: '8px' }} /> 
                  Options d&apos;export
                </Typography>
              </AccordionSummary>
              <AccordionDetails>
                <FormGroup>
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.exportOptions?.allowPdfExport || false}
                        onChange={handleExportOptionChange('allowPdfExport')}
                      />
                    }
                    label="Autoriser l&apos;export PDF"
                  />
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.exportOptions?.allowPrint || false}
                        onChange={handleExportOptionChange('allowPrint')}
                      />
                    }
                    label="Autoriser l&apos;impression"
                  />
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.exportOptions?.allowDataExport || false}
                        onChange={handleExportOptionChange('allowDataExport')}
                      />
                    }
                    label="Autoriser l&apos;export des données"
                  />
                </FormGroup>
              </AccordionDetails>
            </Accordion>
          </Grid>

          {/* Notifications */}
          <Grid item xs={12}>
            <Accordion>
              <AccordionSummary expandIcon={<ExpandMoreIcon />}>
                <Typography sx={{ display: 'flex', alignItems: 'center' }}>
                  <FontAwesomeIcon icon={faGlobe} style={{ marginRight: '8px' }} /> 
                  Notifications
                </Typography>
              </AccordionSummary>
              <AccordionDetails>
                <FormGroup>
                  <FormControlLabel
                    control={
                      <Switch
                        checked={formulaire.notifications || false}
                        onChange={(e) => onChange({ notifications: e.target.checked })}
                      />
                    }
                    label="Activer les notifications par email"
                  />
                </FormGroup>
              </AccordionDetails>
            </Accordion>
          </Grid>
        </Grid>

        <Box sx={{ mt: 3, mb: 2, bgcolor: 'primary.lighter',color: 'primary.main', p: 2, borderRadius: 1 }}>
          <Typography variant="body2" sx={{ display: 'flex', alignItems: 'center' }}>
            <FontAwesomeIcon icon={faInfoCircle} style={{ marginRight: '8px' }} />
            Ces paramètres s&apos;appliquent à l&apos;ensemble du formulaire. Pour configurer des champs individuels, utilisez l&apos;éditeur de propriétés après avoir sélectionné un champ spécifique.
          </Typography>
        </Box>
      </Box>
    </Box>
  );
} 