"use client"

import React, { useEffect } from "react"
import { Box, Button, Grid, Typography } from "@mui/material"
import { useFormik } from "formik"
import { nanoid } from "nanoid"
import { type IBesoinRecrutement, BesoinRecrutementStatut } from "@/shared/types/besoin-recrutement"
import { type ModeType, type TableColumn, TableType, type FormFieldType } from "@/shared/types/common"
import FormField from "@/shared/components/form/form-field"
import { besoinRecrutementFormFields } from "../utils/form-fields-besoin-recrutement"
import FormContainer from "@/layouts/form/form-container"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import EditExportButtons from "@/shared/components/form/edit-export-buttons"
import { besoinRecrutementSchema } from "../utils/besoin-recrutement-schema"
import { _candidatures } from "@/shared/_mock/_candidature"

interface BesoinRecrutementFormProps {
  besoinRecrutement: IBesoinRecrutement
  mode: ModeType
  onSave: (besoinRecrutement: IBesoinRecrutement) => void
  onClose: (forceClose: boolean) => void;
  onEdit: (besoinRecrutement: IBesoinRecrutement) => void
  tableHead: TableColumn[]
  updateTabContent: (tabId: string, newContent: IBesoinRecrutement) => void
  tabId: string
}

export default function BesoinRecrutementForm({
  besoinRecrutement,
  mode,
  onSave,
  onClose,
  onEdit,
  tableHead,
  updateTabContent,
  tabId,
}: BesoinRecrutementFormProps) {
  const isViewMode = mode === "view"
  const isEditMode = mode === "edit"
  const isNewMode = mode === "add"

  const formik = useFormik<IBesoinRecrutement>({
    initialValues: besoinRecrutement,
    validationSchema: besoinRecrutementSchema,
    enableReinitialize: true,
    onSubmit: (values) => {
      if (isNewMode) {
        onSave({
          ...values,
          id: nanoid(),
          dateCreation: new Date().toISOString().split("T")[0],
        })
      } else {
        onSave(values)
      }
      onClose(true)
    },
  })

  useEffect(() => {
    if (updateTabContent && tabId && mode !== "view") {
      const timer = setTimeout(() => {
        updateTabContent(tabId, formik.values)
      }, 300)

      return () => {
        clearTimeout(timer)
      }
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const handleEditClick = () => {
    onEdit(besoinRecrutement)
  }

  const handleValider = () => {
    onSave({
      ...formik.values,
      statut: BesoinRecrutementStatut.EnCours,
    })
    onClose(true)
  }

  const handleMarquerTraite = () => {
    if (!formik.values.candidatureAssocieeId) {
      formik.setFieldError(
        "candidatureAssocieeId",
        "Veuillez sélectionner une candidature avant de marquer comme traité"
      )
      return
    }
    onSave({
      ...formik.values,
      statut: BesoinRecrutementStatut.Traite,
    })
    onClose(true)
  }

  const getCandidatureNom = (id: string | undefined) => {
    if (!id) return "Aucune"
    const candidature = _candidatures.find((c) => c.id === id)
    return candidature ? `${candidature.nom} ${candidature.prenom}` : "Non trouvée"
  }

  // Fonction pour rendre chaque champ du formulaire
  const renderField = (field: FormFieldType<IBesoinRecrutement>) => {
    // Ne pas afficher le champ candidatureAssocieeId si le statut est EnAttente
    if (field.name === "candidatureAssocieeId" && formik.values.statut === BesoinRecrutementStatut.EnAttente) {
      return null
    }

    // Ne pas afficher le champ statut si c'est un nouveau besoin
    if (field.name === "statut" && isNewMode) {
      return null
    }

    return (
      <FormField
        field={field}
        value={formik.values[field.name as keyof IBesoinRecrutement]}
        onChange={(name, value) => formik.setFieldValue(name, value)}
        error={
          formik.touched[field.name as keyof IBesoinRecrutement]
            ? (formik.errors[field.name as keyof IBesoinRecrutement] as string | undefined)
            : undefined
        }
        onBlur={formik.handleBlur}
        isReadOnly={isViewMode}
      />
    )
  }

  return (
    <FormContainer
      titre={isNewMode ? "Nouvelle demande de recrutement" : `Demande de recrutement - ${formik.values.demandeur}`}
    >
      <ConditionalComponent isValid={isViewMode}>
        <EditExportButtons
          onEdit={handleEditClick}
          onExport={() => void 0}
          tooltipTitle={TableType.BesoinRecrutement}
          dataRow={besoinRecrutement}
          tableHead={tableHead}
        />
      </ConditionalComponent>

      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={3}>
          {besoinRecrutementFormFields.map((field) => (
            <Grid item xs={12} md={6} key={String(field.name)}>
              {renderField(field)}
            </Grid>
          ))}
        </Grid>

        <ConditionalComponent isValid={!isViewMode}>
          <Box sx={{ mt: 3, display: "flex", justifyContent: "flex-end", gap: 2 }}>
            <Button variant="outlined" onClick={() => onClose(false)}>
              Fermer
            </Button>
            <Button
              type="button"
              variant="contained"
              onClick={() => {
                formik.validateForm().then((errors) => {
                  if (Object.keys(errors).length === 0) {
                    formik.submitForm()
                  }
                })
              }}
            >
              {isNewMode ? "Créer" : "Enregistrer"}
            </Button>
            <ConditionalComponent isValid={formik.values.statut === BesoinRecrutementStatut.EnAttente}>
              <Button variant="contained" color="primary" onClick={handleValider}>
                Valider
              </Button>
            </ConditionalComponent>
            <ConditionalComponent isValid={formik.values.statut === BesoinRecrutementStatut.EnCours}>
              <Button
                variant="contained"
                color="primary"
                onClick={handleMarquerTraite}
                disabled={!formik.values.candidatureAssocieeId}
              >
                Marquer comme traité
              </Button>
            </ConditionalComponent>
          </Box>
        </ConditionalComponent>
      </form>
    </FormContainer>
  )
}

