import React from 'react';
import {
  TableCell,
  Checkbox,
  IconButton,
  Stack,
  Snackbar,
  Alert,
} from '@mui/material';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faShareNodes, faEdit, faTrash, faEye } from '@fortawesome/free-solid-svg-icons';
import { FormulaireTableRowProps } from '../types';
import { ContratStyledRow } from '@/shared/sections/contract/styles';
import SvgColor from '@/shared/components/svg-color';
import { ActionsIconButton } from '@/shared/theme/css';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import ConfirmDialog from '@/shared/components/dialog/confirmDialog';

export default function FormulaireTableRow({
  row,
  selected,
  onSelectRow,
  handleEdit,
  handleDelete,
  handleView,
  dense,
}: FormulaireTableRowProps) {
  const [deleteDialogOpen, setDeleteDialogOpen] = React.useState(false);
  const [snackbarOpen, setSnackbarOpen] = React.useState(false);
 
  const confirmDelete = () => {
    handleDelete(row.id);
    setDeleteDialogOpen(false);
  };
  
  const cancelDelete = () => {
    setDeleteDialogOpen(false);
  };

  const handleCopyLink = () => {
    navigator.clipboard.writeText(row.lien);
    setSnackbarOpen(true);
  };

  const handleCloseSnackbar = () => {
    setSnackbarOpen(false);
  };

  return (
    <>
      <ContratStyledRow hover onDoubleClick={() => handleView(row)} style={{ cursor: 'pointer' }}>
        <TableCell sx={{ whiteSpace: 'nowrap',position: 'relative', py: dense ? 1 : 2 }}>
            <SvgColor
              src="/assets/icons/ic_border.svg"
              sx={{
                position: 'absolute',
                top: 0,
                left: -4,
                height: '100%',
                color: '#1976d2',
              }}
            />
          {row.nom}
        </TableCell>
        <TableCell sx={{ whiteSpace: 'nowrap', py: dense ? 1 : 2 }}>{row.description}</TableCell>
        <TableCell sx={{ whiteSpace: 'nowrap', py: dense ? 1 : 2 }}>
          <a href={row.lien} target="_blank" rel="noopener noreferrer" style={{ color: '#746CD4' }}>
            {row.lien}
          </a>
        </TableCell>
        <TableCell>
          <Stack direction="row" spacing={1} justifyContent="flex-start">
            <CustomTooltip arrow title="Copier le lien">
              <IconButton
                color="primary"
                onClick={(e) => {
                  e.stopPropagation();
                  handleCopyLink();
                }}
                size="small"
                sx={ActionsIconButton}
              >
                <FontAwesomeIcon icon={faShareNodes} size="sm" />
              </IconButton>
            </CustomTooltip>
            <CustomTooltip arrow title="Modifier">
              <IconButton
                color="info"
                onClick={(e) => {
                  e.stopPropagation();
                  handleEdit(row);
                }}
                size="small"
                sx={ActionsIconButton}
              >
                <FontAwesomeIcon icon={faEdit} size="sm" />
              </IconButton>
            </CustomTooltip>
            <CustomTooltip arrow title="Supprimer">
              <IconButton
                color="error"
                onClick={(e) => {
                  e.stopPropagation();
                  setDeleteDialogOpen(true);
                }}
                size="small"
                sx={ActionsIconButton}
              >
                <FontAwesomeIcon icon={faTrash} size="sm" />
              </IconButton>
            </CustomTooltip>
          </Stack>
        </TableCell>
      </ContratStyledRow>
      
      <ConfirmDialog
        open={deleteDialogOpen}
        title="Confirm Delete"
        content="Are you sure you want to delete this document?"
        onConfirm={confirmDelete}
        onCancel={cancelDelete}
      />
      
      <Snackbar
        open={snackbarOpen}
        autoHideDuration={3000}
        onClose={handleCloseSnackbar}
        anchorOrigin={{ vertical: 'bottom', horizontal: 'center' }}
      >
        <Alert onClose={handleCloseSnackbar} severity="success">
          Lien copié dans le presse-papiers
        </Alert>
      </Snackbar>
    </>
  );
}