import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  Typography,
} from '@mui/material';
import { ICandidature } from '@/shared/types/candidature';
import { pxToRem } from '@/shared/theme/typography';
import { StyledTextField } from '@/shared/theme/css';
import ActionButtons from '@/shared/components/form/buttons-action';
import {  StyledDialog, StyledDialogTitle } from '../../../styles';
import { enqueueSnackbar } from 'notistack';

interface SendEmailModalProps {
  open: boolean;
  onClose: () => void;
  candidature: ICandidature;
  onSend: (subject: string, message: string) => void;
}

export default function SendEmailModal({ open, onClose, candidature, onSend }: SendEmailModalProps) {
  const [subject, setSubject] = useState<string>('');
  const [message, setMessage] = useState<string>('');

  const handleSubjectChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setSubject(event.target.value);
  };

  const handleMessageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setMessage(event.target.value);
  };

  const handleSend = () => {
    if (subject && message) {
      onSend(subject, message);
      handleReset();
      onClose();
      enqueueSnackbar('Email envoyé avec succès', { variant: 'success' });
    }
  };

  const handleReset = () => {
    setSubject('');
    setMessage('');
  };

  return (
    <StyledDialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <StyledDialogTitle>
        Envoi d{"'"}un mail
      </StyledDialogTitle>
      <DialogContent>

        <Typography variant="subtitle2" sx={{ display: 'flex', justifyContent: 'center', mb: 2, fontWeight: (theme) => theme.typography.fontWeightSemiBold, }}>
          Destinataire: {candidature.nom} {candidature.prenom} ({candidature.email})
        </Typography>
        <StyledTextField
          fullWidth
          InputLabelProps={{
            shrink: true,
          }}
          label="Titre"
          value={subject}
          placeholder='Choisissez un titre'
          onChange={handleSubjectChange}
          sx={{ mb: 2, mt: 1 }}
          required
        />
        <StyledTextField
          autoFocus
          margin="dense"
          id="description"
          label="Observation"
          InputLabelProps={{
            shrink: true,
          }}
          multiline
          rows={4}
          fullWidth
          value={message}
          onChange={handleMessageChange}
          placeholder="Ajoutez un message explicatif..."
        />
      </DialogContent>
      <DialogActions>
        <ActionButtons onSave={handleSend} onCancel={onClose} secoundButtonText="Confirmer"  />
      </DialogActions>
    </StyledDialog>
  );
} 