'use client';

import { Box, Stack } from '@mui/material';
import { useCallback, useMemo } from 'react';
import { HEADER } from '@/layouts/config-layout';
import TableControlBar from '@/shared/components/table/table-control-bar';
import { type TableConfig, useTableManager } from '@/hooks/use-table-manager';
import { type TabConfig, useTabsManager } from '@/hooks/use-tabs-manager';
import { TableType } from '@/shared/types/common';
import TabsCustom from '@/shared/components/tabs/tabs-custom';
import TableManager from '@/shared/components/table/table-manager';
import { TABLE_HEAD } from '../utils/table-head';
import type { Candidat } from '@/shared/types/candidat';
import { _candidats, DEFAULT_CANDIDAT_TABLE_HEAD, INITIAL_CANDIDAT_DATA } from '@/shared/_mock/_candidats';
import CandidatTableRow from './salarie-table-row';
import CandidatForm from '@/shared/sections/rh/salarie/view/salarie-form';
import { UnsavedChangesDialog } from '@/shared/components/dialog/UnsavedChangesDialog';
const candidatConfig: TableConfig<Candidat> = {
  initialData: _candidats,
  defaultTableHead: [...DEFAULT_CANDIDAT_TABLE_HEAD],
};

const candidatTabsConfig: TabConfig<Candidat> = {
  type: TableType.SALARIE,
  listTitle: "Liste Salariés",
  newItemTitle: "Nouveau Salarié",
  initialData: INITIAL_CANDIDAT_DATA,
  getItemCode: (candidat) => `${candidat.prenom} ${candidat.nom}`,
};

export default function CandidatView() {
  const {
    tabs,
    activeTab,
    handleTabClose,
    handleTabChange,
    handleTabAdd,
    handleEdit,
    handleView,
    handleCancel,
    handleCopy,
    showConfirmDialog,
    setShowConfirmDialog,
    handleConfirmDialogAction,
    updateTabContent,
  } = useTabsManager<Candidat>(candidatTabsConfig);

  const {
    table,
    filteredData,
    filters,
    tableHead,
    notFound,
    handleFilterChange,
    handleResetFilters,
    handleResetColumns,
    handleColumnsChange,
    handleSave,
    handleStatusChange,
    dataToExport
  } = useTableManager<Candidat>(candidatConfig);

  const activeTabData = useMemo(() => {
    return tabs.find((tab) => tab.id === activeTab);
  }, [tabs, activeTab]);

  const renderRow = useCallback(
    (row: Candidat) => (
      <CandidatTableRow
        key={row.id}
        dense={table.dense}
        row={row}
        handleEdit={handleEdit}
        handleCopy={handleCopy}
        handleDetails={handleView}
        selected={table.selected.includes(row.id)}
        onSelectRow={() => table.onSelectRow(row.id)}
        columns={tableHead.slice(0, -1)}
        onStatusChange={handleStatusChange}
      />
    ),
    [table, handleEdit, handleCopy, handleView, tableHead, handleStatusChange],
  );

  const isList = activeTab === 'list';

  return (
    <Box
      sx={{
        display: 'flex',
        flexDirection: 'column',
        height: `calc(100vh - ${HEADER.H_DESKTOP + 8}px)`,
      }}
    >
      <Stack flexGrow={0} direction="row" alignItems="center" justifyContent="flex-end" spacing={{ xs: 0.5, sm: 1 }}>
        <TableControlBar
          type={TableType.SALARIE}
          handleTabAdd={handleTabAdd}
          activeTab={activeTab}
          onResetFilters={handleResetFilters}
          onResetColumns={handleResetColumns}
          onColumnsChange={handleColumnsChange}
          initialColumns={tableHead.slice(0, -1)}
          onFilters={handleFilterChange}
          filteredData={dataToExport}
        />
      </Stack>

      <TabsCustom
        type={TableType.SALARIE}
        tabs={tabs}
        activeTab={activeTab}
        handleTabChange={handleTabChange}
        handleTabClose={handleTabClose}
      />

      {isList ? (
        <TableManager<Candidat>
          table={table}
          filteredData={filteredData}
          tableHead={tableHead}
          notFound={notFound}
          filters={filters}
          onFilterChange={handleFilterChange}
          renderRow={renderRow}
        />
      ) : (
        <CandidatForm
          key={activeTab}
          candidat={activeTabData?.content as Candidat}
          mode={activeTabData?.mode || 'view'}
          onSave={handleSave}
          onClose={(forceClose) => handleCancel(activeTab, forceClose)}
          onEdit={handleEdit}
          updateTabContent={updateTabContent}
          tabId={activeTab}
        />
      )}
      <UnsavedChangesDialog
        open={showConfirmDialog}
        onClose={() => setShowConfirmDialog(false)}
        onConfirm={handleConfirmDialogAction}
        onCancel={() => setShowConfirmDialog(false)}
      />
    </Box>
  );
}