"use client"
import Checkbox from "@mui/material/Checkbox"
import TableCell from "@mui/material/TableCell"
import SvgColor from "@/shared/components/svg-color"
import type { TableColumn } from "@/shared/types/common"
import TableActionColumn from "@/shared/components/table/table-action-column"
import { type IDemande, PrioriteDemande } from "@/shared/types/demande"
import { DemandeStatus } from "@/shared/types/demande"
import { ContratStyledRow, PriorityChip, StatusChip } from "@/shared/sections/contract/styles"
import { formatDate } from "@/utils/format-time"
import { Chip, IconButton, Tooltip, Box } from "@mui/material"
import { Download, AttachFile } from "@mui/icons-material"
import StatusMenu from "@/shared/components/table/StatusMenu"
import { TableConfig, useTableManager } from "@/hooks/use-table-manager"
import { _demandes, DEFAULT_DEMANDE_TABLE_HEAD } from "@/shared/_mock/_demandes"
import PriorityMenu from "@/shared/components/table/PriorityMenu"
import { Condiment } from "next/font/google"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"

type Props = {
  handleEdit: (demande: IDemande) => void
  handleDetails: (demande: IDemande) => void
  handleCopy: (demande: IDemande) => void
  selected: boolean
  row: IDemande
  dense: boolean
  onSelectRow: VoidFunction
  columns: TableColumn[]
  onStatusChange: (row: IDemande, newStatus: string, field: keyof IDemande) => void;
  onPriorityChange: (row: IDemande, newPriority: string, field: keyof IDemande) => void;
}

const TOOLTIP_TITLES = {
  duplicate: "Copier demande",
  active: "Actif",
  inactive: "Inactif",
  download: "Télécharger",
}

const getStatusColor = (status: DemandeStatus) => {
  switch (status) {
    case DemandeStatus.EN_ATTENTE:
      return {
        color: "warning",
        backgroundColor: "warning.lighter",
        borderColor: "warning.light",
      }
    case DemandeStatus.EN_TRAITEMENT:
      return {
        color: "info",
        backgroundColor: "info.lighter",
        borderColor: "info.light",
      }
    case DemandeStatus.VALIDEE:
      return {
        color: "success",
        backgroundColor: "success.lighter",
        borderColor: "success.light",
      }
    case DemandeStatus.REFUSEE:
      return {
        color: "error",
        backgroundColor: "error.lighter",
        borderColor: "error.light",
      }
    default:
      return {
        color: "default",
        backgroundColor: "grey.200",
        borderColor: "grey.300",
      }
  }
}

const getPrioriteColor = (priorite: PrioriteDemande) => {
  switch (priorite) {
    case PrioriteDemande.BASSE:
      return {
        color: "success",
        backgroundColor: "success.lighter",
        borderColor: "success.light",
      }
    case PrioriteDemande.NORMALE:
      return {
        color: "info",
        backgroundColor: "info.lighter",
        borderColor: "info.light",
      }
    case PrioriteDemande.HAUTE:
      return {
        color: "warning",
        backgroundColor: "warning.lighter",
        borderColor: "warning.light",
      }
    case PrioriteDemande.URGENTE:
      return {
        color: "error",
        backgroundColor: "error.lighter",
        borderColor: "error.light",
      }
    default:
      return {
        color: "default",
        backgroundColor: "grey.200",
        borderColor: "grey.300",
      }
  }
}

const DemandeStatusOptions = [
    { value :DemandeStatus.EN_ATTENTE, label: "EN_ATTENTE" },
    { value: DemandeStatus.EN_TRAITEMENT, label: "EN_TRAITEMENT" },
    { value: DemandeStatus.VALIDEE, label: "VALIDEE" },
    { value: DemandeStatus.REFUSEE, label: "REFUSEE" },
    { value: DemandeStatus.EMPTY, label: "vide" }
  ];
const demandePriorityOptions = [
    { value :PrioriteDemande.BASSE, label: "BASSE" },
    { value: PrioriteDemande.NORMALE, label: "NORMALE" },
    { value: PrioriteDemande.HAUTE, label: "HAUTE" },
    { value: PrioriteDemande.URGENTE, label: "URGENTE" },
  ];

  const demandeConfig: TableConfig<IDemande> = {
    initialData: _demandes,
    defaultTableHead: DEFAULT_DEMANDE_TABLE_HEAD,
  }

export default function DemandeTableRow({
  dense,
  row,
  handleEdit,
  handleDetails,
  handleCopy,
  selected,
  onSelectRow,
  columns,
  onStatusChange,
  onPriorityChange
}: Props) {
    const {handleStatusChange } = useTableManager<IDemande>(demandeConfig)
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "dateCreation":
      case "dateMiseAJour":
      case "dateExpiration":
        return formatDate(value)
        case "status":
            return (
                <StatusMenu
                    value={value}
                    options={DemandeStatusOptions}
                    onChange={(newStatus) => onStatusChange(row, newStatus, 'status')}
                    chipComponent={StatusChip}
                    disabled
                />
            )
      case "priorite":
         return (
                <StatusMenu
                    value={value}
                    options={demandePriorityOptions}
                    onChange={(newStatus) => onStatusChange(row, newStatus, 'priorite')}
                    chipComponent={StatusChip}
                    disabled
                />
            )

      default:
        return value
    }
  }

  const handleDownload = () => {
    console.log("Downloading document for request:", row.numeroDemande)
    // Implement download functionality here
  }

  return (
    <ContratStyledRow hover selected={selected} style={{ cursor: "pointer" }}>
      {columns
        .filter((column) => column.id !== "isArchive")
        .map((column: any,index:number) => (
          <TableCell
            onDoubleClick={() => handleDetails(row)}
            key={column.id}
            align={column.align}
            sx={{ whiteSpace: "nowrap", py: dense ? 0 : 2,...(index === 0 && {position: "relative"}) }}
          >
            <ConditionalComponent isValid={index === 0}>
              <SvgColor
                src="/assets/icons/ic_border.svg"
                sx={{
                  position: "absolute",
                  top: 0,
                  left: -4,
                  height: "100%",
                  color: row.fichierAttache ? "primary.main" : "info.main",
                }}
              />
            </ConditionalComponent>
           
            {renderCellContent(column.id, row[column.id as keyof IDemande])}
          </TableCell>
        ))}

      <TableCell
        sx={{
          px: 1,
          whiteSpace: "nowrap",
          py: 0,
          gap: 0.5,
        }}
      >
        <ConditionalComponent isValid={row.status === DemandeStatus.REFUSEE || row.status === DemandeStatus.EN_ATTENTE}>
        <TableActionColumn
            handleEdit={() => handleEdit(row)}
            onToggle={() => void 0 }
            isActive={row.actif}
            tooltipTitles={TOOLTIP_TITLES}
          />
        </ConditionalComponent>
        <ConditionalComponent isValid={row.status === DemandeStatus.VALIDEE || row.status === DemandeStatus.EN_TRAITEMENT}>
        <TableActionColumn
            onToggle={() => void 0 }
            isActive={row.actif}
            tooltipTitles={TOOLTIP_TITLES}
          />
        </ConditionalComponent>
          
      </TableCell>
    </ContratStyledRow>
  )
}

