"use client"

import { Checkbox, TableCell, Chip, IconButton, Tooltip } from "@mui/material"
import SvgColor from "@/shared/components/svg-color"
import type { TableColumn } from "@/shared/types/common"
import { type IEmail, EmailStatus } from "@/shared/types/email"
import { ContratStyledRow, StatusChip } from "@/shared/sections/contract/styles"
import { formatDateTime } from "@/utils/format-time"
import { Refresh as RefreshIcon } from "@mui/icons-material"
import StatusMenu from "@/shared/components/table/StatusMenu"
import ConditionalComponent from '../../../../components/table/ConditionalComponent';

type Props = {
  handleViewDetails: (email: IEmail) => void
  handleResend: (email: IEmail) => void
  selected: boolean
  row: IEmail
  dense: boolean
  onSelectRow: VoidFunction
  columns: TableColumn[],
   onStatusChange: (row: IEmail, newStatus: string, field: keyof IEmail) => void;
}

const EmailStatusOptions = [
    { value :EmailStatus.EN_ATTENTE, label: "EN_ATTENTE" },
    { value: EmailStatus.ENVOYE, label: "ENVOYE" },
    { value: EmailStatus.ECHEC, label: "ECHEC" },
  ];

export default function EmailTableRow({
  dense,
  row,
  handleViewDetails,
  handleResend,
  selected,
  onSelectRow,
  columns,
  onStatusChange
}: Props) {
  const renderCellContent = (columnId: string, value: any) => {
    switch (columnId) {
      case "dateHeure":
        return formatDateTime(value)
      case "status":
        return (
                    <StatusMenu
                            value={value}
                            options={EmailStatusOptions}
                            onChange={(newStatus) => onStatusChange(row, newStatus, 'status')}
                            chipComponent={StatusChip}
                            disabled
                        />
                    )
      default:
        return value
    }
  }

  const renderStatus = (status: EmailStatus) => {
    let color: "success" | "warning" | "error" | "default" = "default"
    let label = ""

    switch (status) {
      case EmailStatus.ENVOYE:
        color = "success"
        label = "Envoyé"
        break
      case EmailStatus.EN_ATTENTE:
        color = "warning"
        label = "En attente"
        break
      case EmailStatus.ECHEC:
        color = "error"
        label = "Échec"
        break
      default:
        color = "default"
        label = status
    }

    return (
      <Chip
        label={label}
        size="small"
        color={color}
        variant="outlined"
        sx={{
          fontWeight: "medium",
        }}
      />
    )
  }

  return (
    <ContratStyledRow hover style={{ cursor: "pointer" }}>
      {columns.map((column: any, index: number) => (
        <TableCell
          onDoubleClick={() => handleViewDetails(row)}
          key={column.id}
          align={column.align}
          sx={{ whiteSpace: "nowrap", py: dense ? 0 : 2,
            ...(index === 0 && {
              position: 'relative',
            })
          }}
        >
          <ConditionalComponent isValid={index === 0}>
            <SvgColor
              src="/assets/icons/ic_border.svg"
              sx={{
                position: 'absolute',
                top: 0,
                left: -4,
                height: '100%',
                color: "primary.main",
              }}
            />
          </ConditionalComponent>
          {renderCellContent(column.id, row[column.id as keyof IEmail])}
        </TableCell>
      ))}

<TableCell
        sx={{
          width: 40,
          minWidth: 40,
          maxWidth: 40,
          padding: "0px 4px",
          textAlign: "left",
        }}
      >
        {row.status === EmailStatus.ECHEC && (
          <Tooltip title="Relancer l'envoi">
            <IconButton size="small" onClick={() => handleResend(row)} color="primary">
              <RefreshIcon fontSize="small" />
            </IconButton>
          </Tooltip>
        )}
      </TableCell>
    </ContratStyledRow>
  )
}

