import { memo, useState, useCallback } from 'react';

import Box from '@mui/material/Box';
import { alpha, useTheme } from '@mui/material/styles';
import InputBase from '@mui/material/InputBase';
import IconButton from '@mui/material/IconButton';
import InputAdornment from '@mui/material/InputAdornment';
import Dialog, { dialogClasses } from '@mui/material/Dialog';

import { useBoolean } from '@/hooks';

import Scrollbar from '@/shared/components/scrollbar';
import SearchNotFound from '@/shared/components/search-not-found';

import FontAwesome from '@/shared/components/fontawesome';
import { faMagnifyingGlass } from '@fortawesome/free-solid-svg-icons';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';

// ----------------------------------------------------------------------

function Searchbar() {
  const theme = useTheme();

  const search = useBoolean();

  const [searchQuery, setSearchQuery] = useState('');

  const handleClose = useCallback(() => {
    search.onFalse();
    setSearchQuery('');
  }, [search]);


  const handleSearch = useCallback((event: React.ChangeEvent<HTMLTextAreaElement>) => {
    setSearchQuery(event.target.value);
  }, []);

  const dataFiltered = []

  const notFound = searchQuery && !dataFiltered.length;

  //The function has not been created yet
  const renderItems = () => {
    return [];
  };

  const renderButton = (
    <CustomTooltip title="Rechercher" arrow  >
    <IconButton
      onClick={search.onTrue}
      color="default"
      sx={{
        bgcolor: (theme) => theme.palette.background.default,
        '&:hover': {
          bgcolor: (theme) => theme.palette.action.hover,
        },
      }}
    >
      <FontAwesome icon={faMagnifyingGlass}  />
    </IconButton>
    </CustomTooltip>
  );

  return (
    <>
      {renderButton}

      <Dialog
        fullWidth
        maxWidth="sm"
        open={search.value}
        onClose={handleClose}
        transitionDuration={{
          enter: theme.transitions.duration.shortest,
          exit: 0,
        }}
        PaperProps={{
          sx: {
            mt: 15,
            overflow: 'unset',
            borderRadius: 6,
          },
        }}
        sx={{
          [`& .${dialogClasses.container}`]: {
            alignItems: 'flex-start',
          },
        }}
      >
        <Box
          sx={{
            border: `2px solid ${theme.palette.primary.main}`,
            borderRadius: 6,
            px: 4,
            py: 1,
            mt:0.25,
            mx:0.25,
            transition: 'box-shadow 0.3s ease',
            '&:focus-within': {
              boxShadow: `0 5px 0 ${alpha(theme.palette.primary.main, 0.1)}`,
            },
          }}
        >
          <InputBase
            fullWidth
            autoFocus
            placeholder="Recherchez et accédez vite"
            value={searchQuery}
            onChange={handleSearch}
            startAdornment={
              <InputAdornment
                position="start"
                sx={{
                  marginRight: 3,
                }}
              >
                <FontAwesome icon={faMagnifyingGlass} width={24} sx={{ color: 'text.primary' }} />
              </InputAdornment>
            }
            inputProps={{
              sx: { typography: 'h6' },
            }}
          />
        </Box>

        <Scrollbar sx={{ p: 3, pt: 2, height: 400 }}>
          {notFound ? (
            <SearchNotFound query={searchQuery} sx={{ py: 10, boxShadow: 'none' }} />
          ) : (
            renderItems()
          )}
        </Scrollbar>
      </Dialog>
    </>
  );
}

export default memo(Searchbar);
