import { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  SelectChangeEvent,
} from '@mui/material';

export interface StatusOption {
  value: string;
  label: string;
}

export interface StatusChangeDialogProps<T extends string = string> {
  open: boolean;
  title?: string;
  currentStatus: T;
  statusOptions: StatusOption[];
  onClose: () => void;
  onConfirm: (newStatus: T) => void;
}

export default function StatusChangeDialog<T extends string = string>({
  open,
  title = 'Changer le statut',
  currentStatus,
  statusOptions,
  onClose,
  onConfirm,
}: StatusChangeDialogProps<T>) {
  const [selectedStatus, setSelectedStatus] = useState<T>(currentStatus);

  const handleStatusChange = (event: SelectChangeEvent<string>) => {
    setSelectedStatus(event.target.value as T);
  };

  const handleConfirm = () => {
    onConfirm(selectedStatus);
    onClose();
  };

  return (
    <Dialog open={open} onClose={onClose}>
      <DialogTitle>{title}</DialogTitle>
      <DialogContent>
        <FormControl fullWidth sx={{ mt: 2 }}>
          <InputLabel>Statut</InputLabel>
          <Select
            value={selectedStatus}
            label="Statut"
            onChange={handleStatusChange}
          >
            {statusOptions.map((option) => (
              <MenuItem key={option.value} value={option.value}>
                {option.label}
              </MenuItem>
            ))}
          </Select>
        </FormControl>
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Annuler</Button>
        <Button onClick={handleConfirm} variant="contained">
          Confirmer
        </Button>
      </DialogActions>
    </Dialog>
  );
} 