'use client';

import type React from 'react';

import { useState, type ReactNode } from 'react';
import { IconButton, Stack, Box, Slider, Typography } from '@mui/material';
import FontAwesome from '@/shared/components/fontawesome';
import { WhiteIconButtonStyle } from '@/shared/theme/css';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import SvgColor from '@/shared/components/svg-color';
import {
  faCircleChevronLeft,
  faCircleChevronRight,
  faPrint,
  faRotate,
} from '@fortawesome/free-solid-svg-icons';
import { alpha, styled } from '@mui/system';
import { lightColor, mainButtonStyle } from '@/shared/components/table/styles';
import TableFilter from '@/shared/components/table/table-filter';
import { TableType, TimeUnit } from '@/shared/types/common';
import { DatePickerPopover } from './date-picker-popover';
import { Dayjs } from 'dayjs';
import ConditionalComponent from '../table/ConditionalComponent';

const CustomSlider = styled(Slider)(({ theme }) => ({
  color: '#8B7FFF',
  height: 4,
  '& .MuiSlider-thumb': {
    height: 16,
    width: 16,
    backgroundColor: '#fff',
    border: '6px solid #8B7FFF',
    boxShadow: `0 0 0 3px white`,
  },
  '& .MuiSlider-track': {
    height: 1,
    borderRadius: 2,
  },
  '& .MuiSlider-rail': {
    height: 3,
    borderRadius: 2,
    backgroundColor: alpha(theme.palette.primary.main, 0.2),
  },
}));

export const ExtendedWhiteIconButtonStyle = {
  ...WhiteIconButtonStyle,
  padding: 0.9,
};

interface ScheduleToolbarProps {
  date: Date;
  onDateChange: (date: Dayjs | Date, timeUnit?: TimeUnit) => void;
  onZoomChange: (zoom: number) => void;
  onFilters: (key: string, value: string, condition: string) => void;
  onResetFilters: () => void;
  initialColumns: { id: string; label: string; type?: string }[];
  type: TableType;
  customAction?: ReactNode;
  timeUnit?: TimeUnit;
  goToPreviousMonth?: () => void;
  goToNextMonth?: () => void;
  isChild?: boolean;
  rightAction?: ReactNode;
}

export function ScheduleToolbar({
  date,
  onDateChange,
  onZoomChange,
  onFilters,
  onResetFilters,
  initialColumns,
  type,
  customAction,
  timeUnit = 'day',
  goToPreviousMonth,
  goToNextMonth,
  isChild = false,
  rightAction
}: ScheduleToolbarProps) {
  const [calendarAnchor, setCalendarAnchor] = useState<null | HTMLElement>(null);
  const [zoomValue, setZoomValue] = useState<number>(50);

  const handlePrev = () => {
    const newDate = new Date(date);
    if (timeUnit === 'day') {
      newDate.setDate(date.getDate() - 1);
    } else if (timeUnit === 'month' && goToPreviousMonth) {
      goToPreviousMonth();
      return;
    } else if (timeUnit === 'year') {
      newDate.setFullYear(date.getFullYear() - 1);
    }
    onDateChange(newDate);
  };

  const handleNext = () => {
    const newDate = new Date(date);
    if (timeUnit === 'day') {
      newDate.setDate(date.getDate() + 1);
    } else if (timeUnit === 'month' && goToNextMonth) {
      goToNextMonth();
      return;
    } else if (timeUnit === 'year') {
      newDate.setFullYear(date.getFullYear() + 1);
    }
    onDateChange(newDate);
  };
  const handleCalendarClick = (event: React.MouseEvent<HTMLElement>) => {
    setCalendarAnchor(event.currentTarget);
  };

  const handleCalendarClose = () => {
    setCalendarAnchor(null);
  };

  const handleZoomChange = (_event: Event, newValue: number | number[]) => {
    const zoom = Array.isArray(newValue) ? newValue[0] : newValue;
    setZoomValue(zoom);
    onZoomChange(zoom);
  };

  const getPrevTooltip = () => {
    switch (timeUnit) {
      case 'day':
        return 'Jour précédent';
      case 'month':
        return 'Mois précédent';
      case 'year':
        return 'Année précédente';
      default:
        return 'Précédent';
    }
  };

  const getNextTooltip = () => {
    switch (timeUnit) {
      case 'day':
        return 'Jour suivant';
      case 'month':
        return 'Mois suivant';
      case 'year':
        return 'Année suivante';
      default:
        return 'Suivant';
    }
  };

  return (
    <Box sx={{ display: 'flex', justifyContent: 'center', pb: 1 }}>
      <Stack direction="row" spacing={1} alignItems="center">
        {customAction ? customAction : <></>}
        <CustomTooltip title="Calendrier" arrow>
          <IconButton
            sx={isChild ? lightColor : ExtendedWhiteIconButtonStyle}
            onClick={handleCalendarClick}
          >
            <SvgColor
              src="/assets/icons/ic_date.svg"
              sx={{
                width: 18,
                height: 18,
              }}
            />
          </IconButton>
        </CustomTooltip>

        {type !== TableType.Driver ? <DatePickerPopover
          open={Boolean(calendarAnchor)}
          anchorEl={calendarAnchor}
          onClose={handleCalendarClose}
          date={date}
          onDateChange={onDateChange}
        /> : <></>}

        {customAction ? (
          <></>
        ) : (
          <CustomTooltip title="Imprimer" arrow>
            <IconButton sx={ExtendedWhiteIconButtonStyle} onClick={() => void 0}>
              <FontAwesome icon={faPrint} width={18} />
            </IconButton>
          </CustomTooltip>
        )}

        <CustomTooltip title={getPrevTooltip()} arrow>
          <IconButton sx={mainButtonStyle} onClick={handlePrev}>
            <FontAwesome width={30} icon={faCircleChevronLeft} />
          </IconButton>
        </CustomTooltip>

        <Box
          sx={{
            height: 35,
            width: 300,
            display: 'flex',
            alignItems: 'center',
            mx: 2,
            bgcolor: isChild ? 'primary.lighter' : 'white',
            borderRadius: 5,
            px: 1,
          }}
        >
          <CustomSlider
            value={zoomValue}
            onChange={handleZoomChange}
            aria-labelledby="continuous-slider"
          />

          <Typography sx={{ marginLeft: 2, color: '#6B7280', fontSize: '0.875rem' }}>
            {zoomValue}%
          </Typography>
        </Box>

        <CustomTooltip title={getNextTooltip()} arrow>
          <IconButton sx={mainButtonStyle} onClick={handleNext}>
            <FontAwesome width={30} icon={faCircleChevronRight} />
          </IconButton>
        </CustomTooltip>

        <CustomTooltip title="Rafraîchir" arrow>
          <IconButton
            sx={isChild ? lightColor : ExtendedWhiteIconButtonStyle}
            onClick={onResetFilters}
          >
            <FontAwesome icon={faRotate} width={18} />
          </IconButton>
        </CustomTooltip>

        <TableFilter
          onResetFilters={onResetFilters}
          onFilters={onFilters}
          initialColumns={initialColumns}
          type={type}
          isChild={isChild}
        />

        <ConditionalComponent isValid={!!rightAction}>
          {rightAction}
        </ConditionalComponent> 
      </Stack>
    </Box>
  );
}
