import Stack from "@mui/material/Stack"
import { alpha } from "@mui/material/styles"
import IconButton from "@mui/material/IconButton"
import ListItemText from "@mui/material/ListItemText"

import { fData } from "@/utils/format-number"

import type { UploadProps } from "./types"
import FileThumbnail, { fileData } from "../file-thumbnail"
import FontAwesome from "../fontawesome"
import { faXmark } from "@fortawesome/free-solid-svg-icons"

// ----------------------------------------------------------------------

export default function MultiFilePreview({ thumbnail, files, onRemove, sx }: UploadProps) {
  return (
    <>
      {files?.map((file) => {
        const { key, name = "", size = 0 } = fileData(file)

        const isNotFormatFile = typeof file === "string"

        if (thumbnail) {
          return (
            <Stack
              key={key}
              component="div"
              alignItems="center"
              display="inline-flex"
              justifyContent="center"
              sx={{
                m: 0.5,
                width: 80,
                height: 80,
                borderRadius: 1.25,
                overflow: "hidden",
                position: "relative",
                border: (theme) => `solid 1px ${alpha(theme.palette.grey[500], 0.16)}`,
                ...sx,
              }}
            >
              <FileThumbnail
                tooltip
                imageView
                file={file}
                sx={{ position: "absolute" }}
                imgSx={{ position: "absolute" }}
              />

              {onRemove ? (
                <IconButton
                  size="small"
                  onClick={() => onRemove(file)}
                  sx={{
                    p: 0.5,
                    top: 4,
                    right: 4,
                    position: "absolute",
                    color: "common.white",
                    bgcolor: (theme) => alpha(theme.palette.grey[900], 0.48),
                    "&:hover": {
                      bgcolor: (theme) => alpha(theme.palette.grey[900], 0.72),
                    },
                  }}
                >
                  <FontAwesome icon={faXmark} width={14} />
                </IconButton>
              ) : <></>}
            </Stack>
          )
        }

        return (
          <Stack
            key={key}
            component="div"
            spacing={2}
            direction="row"
            alignItems="center"
            sx={{
              my: 1,
              py: 1,
              px: 1.5,
              borderRadius: 1,
              border: (theme) => `solid 1px ${alpha(theme.palette.grey[500], 0.16)}`,
              ...sx,
            }}
          >
            <FileThumbnail file={file} />

            <ListItemText
              primary={isNotFormatFile ? file : name}
              secondary={isNotFormatFile ? "" : fData(size)}
              secondaryTypographyProps={{
                component: "span",
                typography: "caption",
              }}
            />

            {onRemove ? (
              <IconButton size="small" onClick={() => onRemove(file)}>
                <FontAwesome icon={faXmark} width={16} />
              </IconButton>
            ) : <></>}
          </Stack>
        )
      })}
    </>
  )
}

