"use client"

import type React from "react"

import { useState } from "react"
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  Box,
  Typography,
  IconButton,
  RadioGroup,
  FormControlLabel,
  Radio,
} from "@mui/material"
import { Close as CloseIcon, PictureAsPdf, Description } from "@mui/icons-material"
import type { IEmailTemplate } from "@/shared/types/email-template"

interface EmailTemplateExportProps {
  open: boolean
  template: IEmailTemplate | null
  onClose: () => void
}

export default function EmailTemplateExport({ open, template, onClose }: EmailTemplateExportProps) {
  const [exportFormat, setExportFormat] = useState<"pdf" | "word">("pdf")

  if (!template) {
    return null
  }

  const handleExportFormatChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setExportFormat(event.target.value as "pdf" | "word")
  }

  const handleDownload = () => {
    console.log(`Téléchargement du modèle au format ${exportFormat}`)

    // Logique de téléchargement selon le format choisi
    if (exportFormat === "pdf") {
      // Simuler un téléchargement PDF
      const blob = new Blob([template.contenu], { type: "application/pdf" })
      const url = URL.createObjectURL(blob)
      const a = document.createElement("a")
      a.href = url
      a.download = `${template.titre}.pdf`
      document.body.appendChild(a)
      a.click()
      document.body.removeChild(a)
      URL.revokeObjectURL(url)
    } else {
      // Simuler un téléchargement Word
      const blob = new Blob([template.contenu], {
        type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      })
      const url = URL.createObjectURL(blob)
      const a = document.createElement("a")
      a.href = url
      a.download = `${template.titre}.docx`
      document.body.appendChild(a)
      a.click()
      document.body.removeChild(a)
      URL.revokeObjectURL(url)
    }

    onClose()
  }

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="sm"
      fullWidth
      PaperProps={{
        sx: {
          borderRadius: 4,
          overflow: "hidden",
        },
      }}
    >
      <DialogTitle
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          backgroundColor: "primary.main",
          color: "white",
          fontWeight: "600",
          padding: "16px 24px",
          borderBottom: (theme) => `1px solid ${theme.palette.divider}`,
        }}
      >
        <Typography variant="h6" sx={{ fontSize: "1.25rem", fontWeight: "bold", color: "white" }}>
        Exporter le modèle d&#39;email
        </Typography>
        <IconButton aria-label="close" onClick={onClose} size="small" sx={{ color: "white" }}>
          <CloseIcon />
        </IconButton>
      </DialogTitle>

      <DialogContent
        dividers
        sx={{
          backgroundColor: "#fafafa",
          padding: "24px",
          display: "flex",
          flexDirection: "column",
          gap: "16px",
        }}
      >
        <Box sx={{ mb: 2 }}>
          <Typography variant="subtitle1" gutterBottom>
            Modèle à exporter: <strong>{template.titre}</strong>
          </Typography>
          <Typography variant="body2" color="text.secondary" gutterBottom>
            {template.description}
          </Typography>
        </Box>

        <Box sx={{ mt: 3 }}>
          <Typography variant="subtitle1" gutterBottom>
          Format d&#39;exportation
          </Typography>
          <RadioGroup value={exportFormat} onChange={handleExportFormatChange}>
            <FormControlLabel
              value="pdf"
              control={<Radio />}
              label={
                <Box sx={{ display: "flex", alignItems: "center" }}>
                  <PictureAsPdf color="error" sx={{ mr: 1 }} />
                  <Typography>PDF</Typography>
                </Box>
              }
            />
            <FormControlLabel
              value="word"
              control={<Radio />}
              label={
                <Box sx={{ display: "flex", alignItems: "center" }}>
                  <Description color="primary" sx={{ mr: 1 }} />
                  <Typography>Word (DOCX)</Typography>
                </Box>
              }
            />
          </RadioGroup>
        </Box>
      </DialogContent>

      <DialogActions sx={{ padding: "16px 24px", backgroundColor: "#fafafa" }}>
        <Button
          onClick={onClose}
          variant="outlined"
          sx={{
            borderRadius: "8px",
            padding: "8px 16px",
            color: "primary.main",
            borderColor: "primary.main",
            "&:hover": {
              borderColor: "primary.dark",
              color: "primary.dark",
            },
          }}
        >
          Annuler
        </Button>
        <Button
          variant="contained"
          onClick={handleDownload}
          sx={{
            borderRadius: "8px",
            padding: "8px 16px",
            backgroundColor: "primary.main",
            "&:hover": {
              backgroundColor: "primary.dark",
            },
            boxShadow: 2,
          }}
        >
          Télécharger
        </Button>
      </DialogActions>
    </Dialog>
  )
}
