
import { ISubmittedRoadmap } from '@/shared/types/driver';
import { IRoadmap, SubmittedRoadMapDTO } from '@/shared/types/roadMap';

import { Dialog, DialogTitle, DialogContent, DialogActions, Button, TextField, Stack, IconButton, Typography, Box } from '@mui/material';
import { Close as CloseIcon } from '@mui/icons-material';
import { useState, useEffect } from 'react';
import { Formik, Field, Form, ErrorMessage } from 'formik';
import * as Yup from 'yup';
import dynamic from 'next/dynamic'; 
import "react-quill/dist/quill.snow.css"; 
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';
import { StyledDialog } from '@/shared/sections/rh/styles';
import { dialogTitleStyles, dialogTitleTextStyles, StyledTextField } from '@/shared/theme/css';
import FontAwesome from '@/shared/components/fontawesome';
import { faRocketchat } from '@fortawesome/free-brands-svg-icons';

const ReactQuill = dynamic(() => import('react-quill'), { ssr: false });

const validationSchema = Yup.object({
  email: Yup.string().email('Veuillez entrer une adresse email valide.').required('L\'email est requis.'),
  subject: Yup.string().required('Le sujet est requis.'),
  message: Yup.string().required('Le message est requis.')
});

export default function SendEmailDialog({
  open,
  onClose,
  roadmap,
}: {
  open: boolean;
  onClose: () => void;
  roadmap: IRoadmap | SubmittedRoadMapDTO;
}) {
  const [email, setEmail] = useState(roadmap.email); 
  const [message, setMessage] = useState('');
  const [subject, setSubject] = useState('');

  useEffect(() => {
    setEmail(roadmap.email); 
  }, [roadmap.email]);

  const handleSubmit = (values: any) => {
    onClose();
  };

  const handleEditorChange = (content: string) => {
    setMessage(content); 
  };

  const modules = {
    toolbar: [
      [{ header: [1, 2, false] }],
      ['bold', 'italic', 'underline', 'strike', 'blockquote'],
      [{ list: 'ordered' }, { list: 'bullet' }],
      [{ color: [] }, { background: [] }],
      ['link', 'image'],
      ['clean'],
    ],
  };

  return (
    <StyledDialog open={open} onClose={onClose} fullWidth>
      <DialogTitle sx={(theme) => dialogTitleStyles(theme)}>
      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
       <FontAwesome icon={faRocketchat} width={20} />
        <Typography variant="h6" sx={dialogTitleTextStyles}>
          Envoyer un Email
        </Typography>
        </Box>
        <IconButton aria-label="close" onClick={onClose} size="small" sx={{ color: 'white' }}>
          <CloseIcon />
        </IconButton>
      </DialogTitle>

      <Formik
        initialValues={{ email, subject, message }}
        validationSchema={validationSchema}
        onSubmit={handleSubmit}
      >
        {({ setFieldValue, values, errors, touched }) => (
          <Form>
            <DialogContent sx={{ paddingTop: '24px' }}>
              <Stack spacing={3}>
              <Field
                  name="email"
                  as={StyledTextField}
                  label="Email"
                  fullWidth
                  error={touched.email && !!errors.email}
                  helperText={touched.email && errors.email}
                />

                <Field
                  name="subject"
                  as={StyledTextField}
                  label="Objet"
                  fullWidth
                  error={touched.subject && !!errors.subject}
                  helperText={touched.subject && errors.subject}
                />
                <Typography variant="subtitle1" gutterBottom>
                  Contenu de l&apos;email
                </Typography>
                <Box
                  sx={{
                    border: values.message ? '1px solid #ddd' : '1px solid red',
                    borderRadius: 1,
                    marginBottom: '8px',
                  }}
                >
                  <ReactQuill
                    theme="snow"
                    value={values.message}
                    onChange={(content) => setFieldValue('message', content)}  
                    modules={modules}
                    style={{ height: '250px', marginBottom: '50px' }}
                  />
                </Box>
                <ConditionalComponent isValid={!!(errors.message && touched.message)}>
                    <div style={{ color: 'red', fontSize: '12px' }}>
                      {errors.message}
                    </div>
                </ConditionalComponent>
              </Stack>
            </DialogContent>

            <DialogActions sx={{ padding: '16px 24px', backgroundColor: '#fafafa' }}>
              <Button
                onClick={onClose}
                variant="outlined"
                sx={{
                  borderRadius: '8px',
                  padding: '8px 16px',
                  color: 'primary.main',
                  borderColor: 'primary.main',
                  '&:hover': {
                    borderColor: 'primary.dark',
                    color: 'primary.dark',
                  },
                }}
              >
                Annuler
              </Button>
              <Button
                type="submit"
                variant="contained"
                sx={{
                  borderRadius: '8px',
                  padding: '8px 16px',
                  backgroundColor: 'primary.main',
                  '&:hover': {
                    backgroundColor: 'primary.dark',
                  },
                }}
              >
                Envoyer
              </Button>
            </DialogActions>
          </Form>
        )}
      </Formik>
    </StyledDialog>
  );
}
