import React, { useState } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
  Typography,
  SelectChangeEvent,
} from '@mui/material';
import { IFormulaire, ICandidature } from '@/shared/types/candidature';
import { _formulaires } from '@/shared/_mock/_candidature';
import {
  StyledFormControl,
  StyledInputLabel,
  StyledMenuItem,
  StyledSelect,
  StyledTextField,
} from '@/shared/theme/css';
import { pxToRem } from '@/shared/theme/typography';
import ActionButtons from '@/shared/components/form/buttons-action';
import { StyledDialogTitle, StyledDialog } from '../../../styles';
import { enqueueSnackbar } from 'notistack';

interface SendFormModalProps {
  open: boolean;
  onClose: () => void;
  candidature: ICandidature;
  onSend: (formulaireId: string, observation: string) => void;
}

export default function SendFormModal({ open, onClose, candidature, onSend }: SendFormModalProps) {
  const [selectedFormulaire, setSelectedFormulaire] = useState<string>('');
  const [observation, setObservation] = useState<string>('');
  const [formulaires] = useState<IFormulaire[]>();

  const handleFormulaireChange = (event: SelectChangeEvent<unknown>) => {
    setSelectedFormulaire(event.target.value as string);
  };

  const handleObservationChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setObservation(event.target.value);
  };

  const handleSend = () => {
    if (selectedFormulaire) {
      onSend(selectedFormulaire, observation);
      handleReset();
      onClose();
      enqueueSnackbar('Formulaire envoyé avec succès', { variant: 'success' });
    }
  };

  const handleReset = () => {
    setSelectedFormulaire('');
    setObservation('');
  };

  return (
    <StyledDialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <StyledDialogTitle >
        Envoyer un Formulaire Supplémentaire
      </StyledDialogTitle>
      <DialogContent>
        <Typography
          variant="subtitle2"
          sx={{
            display: 'flex',
            justifyContent: 'center',
            mb: 2,
            fontWeight: (theme) => theme.typography.fontWeightSemiBold,
          }}
        >
          Candidat: {candidature.nom} {candidature.prenom}
        </Typography>

        <StyledFormControl fullWidth sx={{ mb: 2, mt: 1 }}>
          <StyledInputLabel id="formulaire-select-label">
            Formulaire <span style={{ color: '#FF0000' }}> *</span>
          </StyledInputLabel>
          <StyledSelect
            labelId="formulaire-select-label"
            id="formulaire-select"
            value={selectedFormulaire}
            onChange={handleFormulaireChange}
            required
            displayEmpty
          >
            {formulaires?.map((form) => (
              <StyledMenuItem key={form.id} value={form.id}>
                {form.nom}
              </StyledMenuItem>
            ))}
          </StyledSelect>
        </StyledFormControl>

        <StyledTextField
          fullWidth
          label="Observation"
          InputLabelProps={{
            shrink: true,
          }}
          multiline
          rows={4}
          value={observation}
          onChange={handleObservationChange}
          placeholder="Ajoutez un message explicatif pour le candidat..."
        />
      </DialogContent>
      <DialogActions>
        <ActionButtons onSave={handleSend} onCancel={onClose} secoundButtonText="Confirmer" />
      </DialogActions>
    </StyledDialog>
  );
}
