
import React from 'react';
import {
  TableRow,
  TableCell,
  Button,
  Box,
  Checkbox,
  Tooltip,
  Typography,
  Chip,
} from '@mui/material';
import LogoutIcon from '@mui/icons-material/Logout';
import ConditionalComponent from '@/shared/components/table/ConditionalComponent';


interface SessionData {
  id: string;
  device: {
    type: 'mac' | 'windows' | 'mobile' | 'tablet' | 'other';
    browser: string;
  };
  location: string;
  lastUsed: string;
  lastUsedDate: Date;
  isCurrent?: boolean;
}

interface Column {
  id: string;
  label: string;
  type: string;
}

interface SessionTableRowProps {
  row: SessionData;
  dense: boolean;
  getDeviceIcon: (deviceType: string) => JSX.Element;
  getDeviceDisplay: (device: { type: string; browser: string }) => string;
  getTimeAgo: (date: Date) => string;
  handleDisconnect: (sessionId: string) => void;
  columns: Column[];
  isMobile: boolean;
}

const SessionTableRow = ({
  row,
  dense,
  getDeviceIcon,
  getDeviceDisplay,
  getTimeAgo,
  handleDisconnect,
  columns,
  isMobile,
}: SessionTableRowProps) => {
 
  return (
    <TableRow
      hover
      sx={{
        '&:last-child td, &:last-child th': { border: 0 },
        ...(row.isCurrent && { 
          backgroundColor: (theme) => theme.palette.success.lighter + '30',
          '&:hover': {
            backgroundColor: (theme) => theme.palette.success.lighter + '50',
          }
        })
      }}
    >
      <TableCell padding={dense ? 'none' : 'normal'}>
        <Box sx={{ display: 'flex', alignItems: 'center' }}>
          <Tooltip title={row.device.type.toUpperCase()}>
            {getDeviceIcon(row.device.type)}
          </Tooltip>
          <Box>
            <Typography variant="body2" sx={{ fontWeight: 500 }}>
              {getDeviceDisplay(row.device)}
            </Typography>
            <ConditionalComponent isValid={row.isCurrent ?? false}>
            <Chip
              label="Actuel"
              size="small"
              color="success"
              sx={{ height: 20, fontSize: '0.7rem', ml: 1 }}
            />
          </ConditionalComponent>
          </Box>
        </Box>
      </TableCell>
      
      <ConditionalComponent isValid={!isMobile}>
        <TableCell 
          sx={{ 
            maxWidth: 200, 
            overflow: 'hidden', 
            textOverflow: 'ellipsis', 
            whiteSpace: 'nowrap' 
          }}
        >
          <Typography variant="body2">{row.location}</Typography>
        </TableCell>
      </ConditionalComponent>
      
      <TableCell>
        <Typography variant="body2">{row.lastUsed}</Typography>
      </TableCell>
      
      <TableCell>
        <Typography variant="body2" color="text.secondary">
          {getTimeAgo(row.lastUsedDate)}
        </Typography>
      </TableCell>
      
      <TableCell align="right">
        <Button
       variant="outlined"
          color="secondary"
          onClick={() => handleDisconnect(row.id)}
          startIcon={isMobile ? null : <LogoutIcon />}
          size={isMobile ? "small" : "medium"}
          disabled={row.isCurrent}
          
        >
          {isMobile ? 'Déco.' : 'Déconnecter'}
        </Button>
      </TableCell>
    </TableRow>
  );
};

export default SessionTableRow;