import { EmailStatus, IEmail, IEmailAttachment } from "@/shared/types/email"
import { TableColumn } from "@/shared/types/common"

// Mock email attachments
const mockAttachments: IEmailAttachment[] = [
  {
    id: "1",
    nom: "contrat_travail.pdf",
    taille: "245 KB",
    type: "application/pdf",
    url: "/documents/contrat_travail.pdf",
  },
  {
    id: "2",
    nom: "planning_janvier.xlsx",
    taille: "128 KB",
    type: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
    url: "/documents/planning_janvier.xlsx",
  },
  {
    id: "3",
    nom: "attestation_assurance.pdf",
    taille: "189 KB",
    type: "application/pdf",
    url: "/documents/attestation_assurance.pdf",
  },
  {
    id: "4",
    nom: "fiche_poste.docx",
    taille: "76 KB",
    type: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
    url: "/documents/fiche_poste.docx",
  },
]

// Generate mock email content
const generateEmailContent = (index: number): string => {
  const contents = [
    `<p>Bonjour,</p>
    <p>Veuillez trouver ci-joint votre contrat de travail pour la période du 01/01/2023 au 31/12/2023.</p>
    <p>Merci de nous le retourner signé dans les plus brefs délais.</p>
    <p>Cordialement,<br>Service RH</p>`,
    
    `<p>Cher(e) collaborateur(trice),</p>
    <p>Nous vous informons que votre planning du mois de janvier est maintenant disponible en pièce jointe.</p>
    <p>N'hésitez pas à nous contacter pour toute question.</p>
    <p>Bien cordialement,<br>Service Planification</p>`,
    
    `<p>Bonjour,</p>
    <p>Suite à votre demande, veuillez trouver ci-joint votre attestation d'assurance professionnelle.</p>
    <p>Cordialement,<br>Service Administratif</p>`,
    
    `<p>Bonjour,</p>
    <p>Nous vous transmettons votre fiche de poste mise à jour suite à notre entretien du 15/01/2023.</p>
    <p>Merci d'en prendre connaissance et de nous faire part de vos éventuelles remarques.</p>
    <p>Cordialement,<br>Direction des Ressources Humaines</p>`,
    
    `<p>Cher(e) collaborateur(trice),</p>
    <p>Nous vous rappelons que la réunion mensuelle aura lieu le 28/01/2023 à 14h00 dans la salle de conférence.</p>
    <p>Votre présence est obligatoire.</p>
    <p>Cordialement,<br>La Direction</p>`,
  ]
  
  return contents[index % contents.length]
}

// Generate mock error logs
const generateErrorLog = (): string => {
  const errors = [
    `ERROR [2023-01-15T14:32:18.456Z]: Failed to connect to SMTP server\nConnection timed out after 30000ms\nAttempting retry (1/3)...`,
    `ERROR [2023-01-16T09:45:22.123Z]: Authentication failed\nInvalid credentials provided\nPlease check SMTP configuration`,
    `ERROR [2023-01-17T16:08:37.789Z]: Recipient address rejected\nUser unknown in virtual alias table\nMessage could not be delivered`,
  ]
  
  return errors[Math.floor(Math.random() * errors.length)]
}

// Generate mock emails
export const _emails: IEmail[] = [...Array(20)].map((_, index) => {
  // Generate date within the last 30 days
  const date = new Date()
  date.setDate(date.getDate() - Math.floor(Math.random() * 30))
  
  // Rotate through statuses
  const statuses = [EmailStatus.ENVOYE, EmailStatus.EN_ATTENTE, EmailStatus.ECHEC]
  const status = statuses[index % statuses.length]
  
  // Recipients
  const recipients = [
    { name: "Jean Dupont", email: "jean.dupont@example.com" },
    { name: "Marie Martin", email: "marie.martin@example.com" },
    { name: "Thomas Bernard", email: "t.bernard@example.com" },
    { name: "Sophie Petit", email: "sophie.p@example.com" },
    { name: "Lucas Dubois", email: "l.dubois@example.com" },
  ]
  const recipient = recipients[index % recipients.length]
  
  // Subjects
  const subjects = [
    "Contrat de travail - À signer",
    "Planning du mois de janvier",
    "Attestation d'assurance professionnelle",
    "Mise à jour de votre fiche de poste",
    "Rappel : Réunion mensuelle",
  ]
  
  return {
    id: (index + 1).toString(),
    dateHeure: date.toISOString(),
    destinataire: recipient.name,
    destinataireEmail: recipient.email,
    objet: subjects[index % subjects.length],
    contenu: generateEmailContent(index % subjects.length),
    status,
    pieceJointes: status !== EmailStatus.ECHEC ? [mockAttachments[index % mockAttachments.length]] : undefined,
    logsErreur: status === EmailStatus.ECHEC ? generateErrorLog() : undefined,
  }
})

export const DEFAULT_EMAIL_TABLE_HEAD: TableColumn[] = [
  { id: "objet", label: "Objet", type: "text" },
  { id: "destinataire", label: "Destinataire", type: "text" },
  { id: "dateHeure", label: "Date/Heure", type: "datetime" },
  { id: "status", label: "Statut", type: "text" },
]

export const INITIAL_EMAIL_DATA: IEmail = {
  id: "",
  dateHeure: new Date().toISOString(),
  destinataire: "",
  destinataireEmail: "",
  objet: "",
  contenu: "",
  status: EmailStatus.EN_ATTENTE,
}
