'use client';

import { Box, Table, TableBody, TableCell, TableHead, TableRow, Typography } from '@mui/material';
import { getDate } from 'date-fns';
import { AddButton } from '@/shared/components/table/styles';
import FontAwesome from '@/shared/components/fontawesome';
import { faSquarePlus, faSun } from '@fortawesome/free-solid-svg-icons';
import { StyledTableContainer } from '@/shared/theme/css';
import CustomTooltip from '@/shared/components/tooltips/tooltip-custom';
import { CalendarCell, CalendarHeader, DateNumber, PeriodContainer, VacancePeriodItem } from './styles';
import { CalendarEvent } from '@/hooks/use-calendar';

interface VacanceCalendarProps {
  date: Date;
  handleTabAdd: () => void;
  formattedMonth: string;
  calendarData: {
    daysOfWeek: string[];
    weeks: Date[][];
  };
  isToday: (date: Date) => boolean;
  isCurrentMonth: (date: Date) => boolean;
  getPeriodsForDate: (date: Date) => CalendarEvent[];
  goToPreviousMonth: () => void;
  goToNextMonth: () => void;
  goToToday: () => void;
  zoomLevel: number;
  isChild?: boolean;
}

export default function VacanceCalendar({
  date,
  handleTabAdd,
  formattedMonth,
  calendarData,
  isToday,
  isCurrentMonth,
  getPeriodsForDate,
  zoomLevel,
  isChild = false,
}: VacanceCalendarProps) {
  const { daysOfWeek, weeks } = calendarData;
  const cellHeight = 60 + (zoomLevel / 100) * 100
  const backgroundColor =  isChild ? 'primary.lighter' : undefined
  return (
    <>
      <CalendarHeader sx={{backgroundColor}}>
        <Box flexGrow={1} />
        <Typography
          sx={{
            fontSize: '23px',
            fontWeight: 600,
            color: 'primary.main',
            position: 'absolute',
            left: '50%',
            transform: 'translateX(-50%)',
          }}
        >
          {formattedMonth}
        </Typography>
        <CustomTooltip title={"Ajouter Une Nouvelle période"} arrow>
        <AddButton
          onClick={handleTabAdd}
          variant="contained"
          endIcon={<FontAwesome icon={faSquarePlus} width={18} />}
        >
          Nouvelle période
        </AddButton>
        </CustomTooltip>
      </CalendarHeader>
      <StyledTableContainer sx={{ maxHeight: 'calc(100vh - 200px)' }}>
        <Table stickyHeader size="small">
          <TableHead>
            <TableRow>
              {daysOfWeek.map((day, index) => (
                <TableCell
                  key={index}
                  align="center"
                  sx={{
                    position: 'sticky',
                    fontSize: '12px',
                    py: 0,
                    backgroundColor: isChild ? 'white' : 'primary.lighter' ,
                    border: 'none',
                  }}
                >
                  {day.charAt(0).toUpperCase() + day.slice(1, 3)}
                </TableCell>
              ))}
            </TableRow>
          </TableHead>
          <TableBody>
            {weeks.map((week, weekIndex) => (
              <TableRow key={weekIndex}>
                {week.map((day, dayIndex) => {
                  const dayIsToday = isToday(day);
                  const dayIsCurrentMonth = isCurrentMonth(day);
                  const periods = getPeriodsForDate(day);
                  
                  return (
                    <CalendarCell key={dayIndex} isCurrentMonth={dayIsCurrentMonth}  sx={{ height: `${cellHeight}px`, backgroundColor }} >
                      <DateNumber
                        isToday={dayIsToday}
                        itemColor={periods.length ? periods[0].color : undefined}
                      >
                        {getDate(day)}
                      </DateNumber>
                      <PeriodContainer>
                        {periods.map((period, periodIndex) => (
                          <VacancePeriodItem
                            key={periodIndex}
                            sx={{
                              backgroundColor: period.bgColor,
                              color:period.color,
                              flex: 1,
                              height:
                                periods.length > 1
                                  ? `calc((100% - ${periods.length * 4}px) / ${periods.length})`
                                  : '100%',
                              opacity: 0.8,
                            }}
                          >
                            <Box
                              sx={{
                                position: 'absolute',
                                bottom: 8,
                                left: 8,
                                display: 'flex',
                                alignItems: 'center',
                                gap: 1,
                              }}
                            >
                              <FontAwesome icon={faSun} width={10} />
                              <Typography
                                variant="caption"
                                sx={{
                                  color: 'inherit',
                                  whiteSpace: 'normal',
                                  fontSize: periods.length === 1 ? '13px' : '12px',
                                }}
                              >
                                {period.label}
                              </Typography>
                            </Box>
                          </VacancePeriodItem>
                        ))}
                      </PeriodContainer>
                    </CalendarCell>
                  );
                })}
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </StyledTableContainer>
    </>
  );
}
