"use client"

import { Grid } from "@mui/material"
import { useFormik } from "formik"
import FormField from "@/shared/components/form/form-field"
import type { FormFieldType, ModeType } from "@/shared/types/common"
import EditExportButtons from "@/shared/components/form/edit-export-buttons"
import ActionButtons from "@/shared/components/form/buttons-action"
import FormContainer from "@/layouts/form/form-container"
import { IVehicleType } from "@/shared/types/flottes/vehicule"
import { vehicleTypeFormFields } from "../utils/form-fields-vehicle-type"
import { vehicleTypeSchema } from "../utils/vehicle-type-schema"
import ConditionalComponent from "@/shared/components/table/ConditionalComponent"
import { useEffect } from "react"

interface VehicleTypeFormProps {
  vehicleType: IVehicleType
  mode: ModeType
  onSave: (updatedVehicleType: IVehicleType) => void
  onEdit: (updatedVehicleType: IVehicleType) => void
  onClose: () => void
  setHasUnsavedChanges?: (value: boolean) => void
  updateTabContent?: (tabId: string, newContent: IVehicleType) => void
  tabId?: string
}

export default function VehicleTypeForm({ vehicleType, mode, onSave, onClose, onEdit, setHasUnsavedChanges, updateTabContent, tabId }: VehicleTypeFormProps) {
  const isReadOnly = mode === "view"

  const formik = useFormik<IVehicleType>({
    initialValues: {
      ...vehicleType,
      isActive: vehicleType.isActive ?? true,
    },
    validationSchema: vehicleTypeSchema,
    onSubmit: (values) => {
      onSave(values)
      if (setHasUnsavedChanges) setHasUnsavedChanges(false)
      onClose()
    },
  })

  useEffect(() => {
    if (updateTabContent && tabId && (mode !== "view")) {
      updateTabContent(tabId, formik.values)
    }
  }, [formik.values, updateTabContent, tabId, mode])

  const renderField = (field: FormFieldType<IVehicleType>) => (
    <FormField
      field={field}
      value={formik.values[field.name as keyof IVehicleType]}
      onChange={(name, value) => formik.setFieldValue(name, value)}
      error={
        Boolean(formik.touched[field.name as keyof IVehicleType]) &&
        formik.errors[field.name as keyof IVehicleType]
          ? String(formik.errors[field.name as keyof IVehicleType])
          : undefined
      }
      onBlur={formik.handleBlur}
      isReadOnly={isReadOnly}
    />
  )

  return (
    <FormContainer titre="Type de véhicule">
            <ConditionalComponent isValid={isReadOnly}>
            <EditExportButtons
              onEdit={() => onEdit(vehicleType)}
              tooltipTitle="le type de véhicule"
            />
            </ConditionalComponent>

      <form onSubmit={formik.handleSubmit}>
        <Grid container spacing={5}>
          {vehicleTypeFormFields.map((field) => (
            <Grid item xs={12} sm={4} key={field.name}>
              {renderField(field)}
            </Grid>
          ))}
   <ConditionalComponent isValid={!isReadOnly}>
            <Grid item xs={12}>
              <ActionButtons onSave={formik.submitForm} onCancel={onClose} mode={mode} />
            </Grid>
            </ConditionalComponent>
        </Grid>
      </form>
    </FormContainer>
  )
}
